﻿
//
//    Copyright © 2010, 2011 Thomas C. McDermott, N5EG
//    This file is part of ABCDmatrix - the 2-Port Network Calculator program.
//
//    ABCDmatrix is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    ABCDmatrix is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with ABCDmatrix, if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//


using System;

namespace ABCDmatrix
{
    /// <summary>
    /// Hold a V/I pair with the state of the node voltages and currents that exist
    /// between all tiles and at the end and beginning of the list.
    /// </summary>
    [Serializable]
    public class NodeState
    {
        /// <summary>
        /// (Complex) voltage at the node (to ground).
        /// </summary>
        public Complex V;

        /// <summary>
        /// (Complex) current through the node.
        /// </summary>
        public Complex I;

        /// <summary>
        /// Default constructor
        /// </summary>
        public NodeState()
        {
            V = new Complex();
            I = new Complex();
        }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="volt"></param>
        /// <param name="amp"></param>
        public NodeState(Complex volt, Complex amp)
        {
            V = new Complex(volt);
            I = new Complex(amp);
        }

        /// <summary>
        /// Copy constructor
        /// </summary>
        /// <param name="copyfrom"></param>
        public NodeState(NodeState copyfrom)
        {
            this.V = copyfrom.V;
            this.I = copyfrom.I;
        }

        /// <summary>
        /// Multiply parameter set (2x2) by node state (1x2).
        /// </summary>
        /// <param name="p">ParameterSet</param>
        /// <param name="n">NodeState</param>
        /// <returns>NodeState product</returns>
        public static NodeState operator *(ParameterSet p, NodeState n)
        {
            NodeState r = new NodeState();

            r.V = p.P11A * n.V + p.P12B * n.I;
            r.I = p.P21C * n.V + p.P22D * n.I;

            return r;
        }

        public override bool Equals(object obj)
        {
            // Normally object.Equals tests for reference equality
            // We override to test for value equality

            if (obj == null || this.GetType() != obj.GetType())
                return false;

            NodeState n = (NodeState)obj;
            return (this.V == n.V && this.I == n.I);
        }

        public override int GetHashCode()
        {
            return V.GetHashCode() ^ I.GetHashCode();		// bitwise exclusive-or
        }

        /// <summary>
        /// Compare two NodeStates for equality
        /// </summary>
        /// <param name="n1"></param>
        /// <param name="n2"></param>
        /// <returns>True if both n1 and n2 are equal, or they refer to the same object.</returns>
        public static Boolean operator ==(NodeState n1, NodeState n2)
        {
            return Object.Equals(n1, n2);
        }

        /// <summary>
        /// Compare two NodeStates for inequality
        /// </summary>
        /// <param name="n1"></param>
        /// <param name="n2"></param>
        /// <returns>True if n1 and n2 are not equal, or they refer to different objects.</returns>
        public static Boolean operator !=(NodeState n1, NodeState n2)
        {
            return !Object.Equals(n1, n2);
        }


    }
}
