﻿
//
//    Copyright © 2010 Thomas C. McDermott, N5EG
//    This file is part of ABCDmatrix - the 2-Port Network Calculator program.
//
//    ABCDmatrix is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    ABCDmatrix is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with ABCDmatrix, if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//


using System;
using System.Drawing;
using System.Windows.Forms;


namespace ABCDmatrix
{
    public partial class TileInputForm : Form
    {
        Tile tile;
        private Rectangle rect;
        private CableSet cable;
        private static int schematicWidth = 240;
        private static int schematicHeight = 160;

        public TileInputForm(Tile inputtile)
        {
            InitializeComponent();
            tile = inputtile;
            rect = new Rectangle(200, 290,      // where tile graphic is drawn on the form
                schematicWidth, schematicHeight);

            cable = new CableSet();             // set User cable type, no parameter values

        }

        /// <summary>
        /// User clicked OK - validate numeric values if any, and set selection in the tile
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OKClick(object sender, EventArgs e)
        {
            // Set type and parameter styles for the new tile

            if (ABCDButton.Checked)
                tile.BlockType = NetworkType.ABCDparam;
            if (SButton.Checked)
                tile.BlockType = NetworkType.Sparam;
            if (ZButton.Checked)
                tile.BlockType = NetworkType.Zparam;
            if (YButton.Checked)
                tile.BlockType = NetworkType.Yparam;

            if (shuntYseriesbutton.Checked)
                tile.BlockType = NetworkType.ShuntYserRLC;
            if (shuntYparallelbutton.Checked)
                tile.BlockType = NetworkType.ShuntYparRLC;
            if (seriesZseriesbutton.Checked)
                tile.BlockType = NetworkType.SeriesZserRLC;
            if (seriesZparallelbutton.Checked)
                tile.BlockType = NetworkType.SeriesZparRLC;

            if (transformerbutton.Checked)
                tile.BlockType = NetworkType.Transformer;
            if (seriesTLbutton.Checked)
                tile.BlockType = NetworkType.SeriesTL;
            if (pibutton.Checked)
                tile.BlockType = NetworkType.PiNet;
            if (teebutton.Checked)
                tile.BlockType = NetworkType.TeeNet;

            if (seriesstubButton.Checked)
                tile.BlockType = NetworkType.SeriesStub;
            if (shuntstubButton.Checked)
                tile.BlockType = NetworkType.ShuntStub;

            if (FixedButton.Checked)
                tile.ParamType = ParameterType.Fixed;
            if (VariableButton.Checked)
                tile.ParamType = ParameterType.VariableFreq;

            if (EmptyradioButton.Checked)
                tile.BlockType = NetworkType.Empty;


            if (EmptyradioButton.Checked)   // no parameters needed
            {
                tile.Select = true;
                DialogResult = DialogResult.OK;
                return;
            }

            if (ABCDButton.Checked ||       // get these parameters from other entry form
                SButton.Checked ||
                ZButton.Checked ||
                YButton.Checked )  
            {
                if (ForcecheckBox.Checked == true) // if we need to force the input current, get forcing ratios
                {
                    Double I1I2Mag, I1I2Ang;

                    try
                    {
                        I1I2Mag = Convert.ToDouble(I1I2MagtextBox.Text);
                        I1I2Ang = Convert.ToDouble(I1I2AngletextBox.Text);
                    }
                    catch (FormatException f)
                    {
                        MessageBox.Show("Invalid numeric format for forcing ratio parameter(s)", f.Message,
                            MessageBoxButtons.OK, MessageBoxIcon.Error);
                        return;
                    }

                    // convert mag from dB to linear
                    Double mag = Math.Pow(10, I1I2Mag / 20);
                    // convert degrees to radians
                    Double angle = Math.PI / 180 * I1I2Ang;

                    // Note the sign flip, below. I2 is into the tile, conventially I1 is out of
                    // the tile (thus into the previous tile). In order to make the user parameter
                    // entry imply both currents are into the tile, the sign of I1 needs to be flipped.

                    tile.I1I2ratio.real = -mag * Math.Cos(angle);  // Current forcing ratio.real
                    tile.I1I2ratio.imag = -mag * Math.Sin(angle);  // Current forcing ratio.imag
                    tile.ForceI1I2 = true;
                }

                tile.Select = true;
                DialogResult = DialogResult.OK;
                return;
            }

            // validate numeric input parameters for component types

            double line1, line2, line3, line4, line5, line6, line7, line8, line9;
            
            try    // all other formats return at least one value
            {
                line1 = Convert.ToDouble(R_Zreal_TrRatioTextBox.Text); 
            }
            catch (FormatException f)
            {
                MessageBox.Show("Invalid numeric format", f.Message,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            tile.R_Zreal_TrRatio = line1;             //R or Zo.real or Transformer N:1

            if (shuntYseriesbutton.Checked ||
                shuntYparallelbutton.Checked ||
                seriesZseriesbutton.Checked ||
                seriesZparallelbutton.Checked ||
                seriesTLbutton.Checked ||
                seriesstubButton.Checked ||
                shuntstubButton.Checked)
            {
                try   // these formats return the other two values as well
                {
                    line2 = Convert.ToDouble(LuHy_VFTextBox.Text);  
                    line3 = Convert.ToDouble(Cpf_LenMetersTextBox.Text); 
                }
                catch (FormatException f)
                {
                    MessageBox.Show("Invalid numeric format for parameter(s)", f.Message,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
                tile.LuHy_VF = line2;    // L uHy or VF
                tile.Cpf_LenMeters = line3;    // C pF.
            }

            if (transformerbutton.Checked)
           {
                try   // these formats return the other two values as well
                {
                    line2 = Convert.ToDouble(LuHy_VFTextBox.Text);
                    line3 = Convert.ToDouble(Cablek1FreqTransfKTextBox.Text);
                }
                catch (FormatException f)
                {
                    MessageBox.Show("Invalid numeric format for parameter(s)", f.Message,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
                tile.LuHy_VF = line2;    // primary inductance uHy
                tile.Cablek1Freq_TransfK = line3;    // transformer coupling coefficient.
            }

            
            if (seriesTLbutton.Checked || seriesstubButton.Checked || shuntstubButton.Checked)
            {
                try // transmission line has 6 parameters
                {
                    line4 = Convert.ToDouble(Cablek1FreqTransfKTextBox.Text);  
                    line5 = Convert.ToDouble(CableZimagTextBox.Text);  
                    line8 = Convert.ToDouble(Cablek2sqrtFreqTextBox.Text);
                }
                catch (FormatException f)
                {
                    MessageBox.Show("Invalid numeric format for series TL parameter(s)", f.Message,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
                tile.CableZimag = line5;  // Cable Zo.imag
                tile.Cablek1Freq_TransfK = line4;   // loss(Freq)
                tile.Cablek2sqrtFreq = line8;   // loss sqrt(freq)
            }

            // param6, param7, and param9 are stub termination components
            if (seriesstubButton.Checked || shuntstubButton.Checked)
            {
                try // transmission line has 5 parameters
                {
                    line6 = Convert.ToDouble(StubTermRTextBox.Text);
                    line7 = Convert.ToDouble(StubTermLTextBox.Text);
                    line9 = Convert.ToDouble(StubTermCTextBox.Text);
                }
                catch (FormatException f)
                {
                    MessageBox.Show("Invalid numeric format for stub termination parameter(s)", f.Message,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
                tile.StubTermR = line6;         // stub termination R
                tile.StubTermLuHy = line7;      // stub termination L, uHy
                tile.StubTermCpF = line9;       // stub termination C, pF.
            }

            tile.Select = true;
            DialogResult = DialogResult.OK;
            return;
        }

        /// <summary>
        /// User changed which network type radio button is checked.
        /// Force form redraw so new graphic and text values show immediately.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void TypeToggled(object sender, EventArgs e)
        {
            Refresh(); 
        }

        /// <summary>
        /// Paint form with tilebitmap and text/input box updates
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void PaintForm(object sender, PaintEventArgs e)
        {
            if (EmptyradioButton.Checked)
            {
                e.Graphics.DrawImage(tilebitmaps.Empty, rect);
                R_Zreal_TrRatioLabel.Text = "";
                LuHy_VFLabel.Text = "";
                Cpf_LenMetersLabel.Text = "";
                Cablek2sqrtFreqLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableZimagLabel.Text = "";
                StubTermRLabel.Text = "";
                StubTermLLabel.Text = "";
                StubTermCLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableTypelabel.Text = "";
                R_Zreal_TrRatioTextBox.Enabled = false;
                LuHy_VFTextBox.Enabled = false;
                Cpf_LenMetersTextBox.Enabled = false;
                Cablek1FreqTransfKTextBox.Enabled = false;
                CableZimagTextBox.Enabled = false;
                StubTermRTextBox.Enabled = false;
                StubTermLTextBox.Enabled = false;
                StubTermCTextBox.Enabled = false;
                Cablek2sqrtFreqTextBox.Enabled = false;
                CablecomboBox.Enabled = false;
                ValFreqGroupBox.Enabled = false;
                ParamValueGroupBox.Enabled = false;
                ForcegroupBox.Enabled = false;
                ForcecheckBox.Checked = false;
            }

            if (ABCDButton.Checked)
            {
                e.Graphics.DrawImage(tilebitmaps.ABCDmatrix, rect);
                R_Zreal_TrRatioLabel.Text = "";
                LuHy_VFLabel.Text = "";
                Cpf_LenMetersLabel.Text = "";
                Cablek2sqrtFreqLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableZimagLabel.Text = "";
                StubTermRLabel.Text = "";
                StubTermLLabel.Text = "";
                StubTermCLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableTypelabel.Text = "";
                R_Zreal_TrRatioTextBox.Enabled = false;
                LuHy_VFTextBox.Enabled = false;
                Cpf_LenMetersTextBox.Enabled = false;
                Cablek1FreqTransfKTextBox.Enabled = false;
                CableZimagTextBox.Enabled = false;
                StubTermRTextBox.Enabled = false;
                StubTermLTextBox.Enabled = false;
                StubTermCTextBox.Enabled = false;
                Cablek2sqrtFreqTextBox.Enabled = false;
                CablecomboBox.Enabled = false;
                ValFreqGroupBox.Enabled = true;
                ParamValueGroupBox.Enabled = false;
                ForcegroupBox.Enabled = true;
            }

            if (SButton.Checked)
            {
                e.Graphics.DrawImage(tilebitmaps.Smatrix, rect);
                R_Zreal_TrRatioLabel.Text = "";
                LuHy_VFLabel.Text = "";
                Cpf_LenMetersLabel.Text = "";
                Cablek2sqrtFreqLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableZimagLabel.Text = "";
                StubTermRLabel.Text = "";
                StubTermLLabel.Text = "";
                StubTermCLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableTypelabel.Text = "";
                R_Zreal_TrRatioTextBox.Enabled = false;
                LuHy_VFTextBox.Enabled = false;
                Cpf_LenMetersTextBox.Enabled = false;
                Cablek1FreqTransfKTextBox.Enabled = false;
                CableZimagTextBox.Enabled = false;
                StubTermRTextBox.Enabled = false;
                StubTermLTextBox.Enabled = false;
                StubTermCTextBox.Enabled = false;
                Cablek2sqrtFreqTextBox.Enabled = false;
                CablecomboBox.Enabled = false;
                ValFreqGroupBox.Enabled = true;
                ParamValueGroupBox.Enabled = false;
                ForcegroupBox.Enabled = true;
            }

            if (ZButton.Checked)
            {
                e.Graphics.DrawImage(tilebitmaps.Zmatrix, rect);
                R_Zreal_TrRatioLabel.Text = "";
                LuHy_VFLabel.Text = "";
                Cpf_LenMetersLabel.Text = "";
                Cablek2sqrtFreqLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableZimagLabel.Text = "";
                StubTermRLabel.Text = "";
                StubTermLLabel.Text = "";
                StubTermCLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableTypelabel.Text = "";
                R_Zreal_TrRatioTextBox.Enabled = false;
                LuHy_VFTextBox.Enabled = false;
                Cpf_LenMetersTextBox.Enabled = false;
                Cablek1FreqTransfKTextBox.Enabled = false;
                CableZimagTextBox.Enabled = false;
                StubTermRTextBox.Enabled = false;
                StubTermLTextBox.Enabled = false;
                StubTermCTextBox.Enabled = false;
                Cablek2sqrtFreqTextBox.Enabled = false;
                CablecomboBox.Enabled = false;
                ValFreqGroupBox.Enabled = true;
                ParamValueGroupBox.Enabled = false;
                ForcegroupBox.Enabled = true;
            }

            if (YButton.Checked)
            {
                e.Graphics.DrawImage(tilebitmaps.Ymatrix, rect);
                R_Zreal_TrRatioLabel.Text = "";
                LuHy_VFLabel.Text = "";
                Cpf_LenMetersLabel.Text = "";
                Cablek2sqrtFreqLabel.Text = "";
                CableZimagLabel.Text = "";
                StubTermRLabel.Text = "";
                StubTermLLabel.Text = "";
                StubTermCLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableTypelabel.Text = "";
                R_Zreal_TrRatioTextBox.Enabled = false;
                LuHy_VFTextBox.Enabled = false;
                Cpf_LenMetersTextBox.Enabled = false;
                Cablek1FreqTransfKTextBox.Enabled = false;
                CableZimagTextBox.Enabled = false;
                StubTermRTextBox.Enabled = false;
                StubTermLTextBox.Enabled = false;
                StubTermCTextBox.Enabled = false;
                Cablek2sqrtFreqTextBox.Enabled = false;
                CablecomboBox.Enabled = false;
                ValFreqGroupBox.Enabled = true;
                ParamValueGroupBox.Enabled = false;
                ForcegroupBox.Enabled = true;
            }


            if (shuntYseriesbutton.Checked || shuntYparallelbutton.Checked)
            {
                e.Graphics.DrawImage(tilebitmaps.Shunt_Y, rect);
                R_Zreal_TrRatioLabel.Text = "Resistance, Ohms";
                LuHy_VFLabel.Text = "Inductance, uHy";
                Cpf_LenMetersLabel.Text = "Capacitance, pF";
                Cablek2sqrtFreqLabel.Text = "";
                CableZimagLabel.Text = "";
                StubTermRLabel.Text = "";
                StubTermLLabel.Text = "";
                StubTermCLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableTypelabel.Text = "";
                R_Zreal_TrRatioTextBox.Enabled = true;
                LuHy_VFTextBox.Enabled = true;
                Cpf_LenMetersTextBox.Enabled = true;
                Cablek1FreqTransfKTextBox.Enabled = false;
                CableZimagTextBox.Enabled = false;
                StubTermRTextBox.Enabled = false;
                StubTermLTextBox.Enabled = false;
                StubTermCTextBox.Enabled = false;
                Cablek2sqrtFreqTextBox.Enabled = false;
                CablecomboBox.Enabled = false;
                ValFreqGroupBox.Enabled = false;
                ParamValueGroupBox.Enabled = true;
                ForcegroupBox.Enabled = false;
                ForcecheckBox.Checked = false;
            }

            if (seriesTLbutton.Checked)
            {
                e.Graphics.DrawImage(tilebitmaps.Series_TL, rect);
                R_Zreal_TrRatioLabel.Text = "Real part of Cable impedance, Ohms";
                LuHy_VFLabel.Text = "Cable Velocity Factor";
                Cpf_LenMetersLabel.Text = "Cable Length, meters";
                Cablek2sqrtFreqLabel.Text = "Cable attenuation dB/meter/Hz [k2]";
                CableZimagLabel.Text = "Imaginary part of Cable impedance, Ohms";
                StubTermRLabel.Text = "";
                StubTermLLabel.Text = "";
                StubTermCLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "Cable attenuation dB/meter/Sqrt Hz [k1]";
                CableTypelabel.Text = "Cable Type";
                R_Zreal_TrRatioTextBox.Enabled = true;
                LuHy_VFTextBox.Enabled = true;
                Cpf_LenMetersTextBox.Enabled = true;
                Cablek1FreqTransfKTextBox.Enabled = true;
                CableZimagTextBox.Enabled = true;
                StubTermRTextBox.Enabled = false;
                StubTermLTextBox.Enabled = false;
                StubTermCTextBox.Enabled = false;
                Cablek2sqrtFreqTextBox.Enabled = true;
                CablecomboBox.Enabled = true;
                ValFreqGroupBox.Enabled = false;
                ParamValueGroupBox.Enabled = true;
                ForcegroupBox.Enabled = false;
                ForcecheckBox.Checked = false;
            }

            if (seriesstubButton.Checked)
            {
                e.Graphics.DrawImage(tilebitmaps.Series_Stub, rect);
                R_Zreal_TrRatioLabel.Text = "Real part of Cable impedance, Ohms";
                LuHy_VFLabel.Text = "Cable Velocity Factor";
                Cpf_LenMetersLabel.Text = "Cable Length, meters";
                Cablek2sqrtFreqLabel.Text = "Cable attenuation dB/meter/Hz [k2].";
                Cablek1FreqTransfKLabel.Text = "Cable attenuation dB/meter/Sqrt Hz [k1].";
                CableZimagLabel.Text = "Imaginary part of Cable impedance, Ohms";
                StubTermRLabel.Text = "Stub termination resistance, Ohms";
                StubTermLLabel.Text = "Stub termination inductance, uHy.";
                StubTermCLabel.Text = "Stub termination capacitance, pF.";
                CableTypelabel.Text = "Cable Type";
                R_Zreal_TrRatioTextBox.Enabled = true;
                LuHy_VFTextBox.Enabled = true;
                Cpf_LenMetersTextBox.Enabled = true;
                Cablek1FreqTransfKTextBox.Enabled = true;
                CableZimagTextBox.Enabled = true;
                StubTermRTextBox.Enabled = true;
                StubTermLTextBox.Enabled = true;
                StubTermCTextBox.Enabled = true;
                Cablek2sqrtFreqTextBox.Enabled = true;
                CablecomboBox.Enabled = true;
                ValFreqGroupBox.Enabled = false;
                ParamValueGroupBox.Enabled = true;
                ForcegroupBox.Enabled = false;
                ForcecheckBox.Checked = false;
            }

            if (shuntstubButton.Checked)
            {
                e.Graphics.DrawImage(tilebitmaps.Shunt_Stub, rect);
                R_Zreal_TrRatioLabel.Text = "Real part of Cable impedance, Ohms";
                LuHy_VFLabel.Text = "Cable Velocity Factor";
                Cpf_LenMetersLabel.Text = "Cable Length, meters";
                Cablek2sqrtFreqLabel.Text = "Cable attenuation dB/meter/Hz [k2].";
                Cablek1FreqTransfKLabel.Text = "Cable attenuation dB/meter/Sqrt Hz [k1].";
                CableZimagLabel.Text = "Imaginary part of Cable impedance, Ohms";
                StubTermRLabel.Text = "Stub termination resistance, Ohms";
                StubTermLLabel.Text = "Stub termination inductance, uHy.";
                StubTermCLabel.Text = "Stub termination capacitance, pF.";
                CableTypelabel.Text = "Cable Type";
                R_Zreal_TrRatioTextBox.Enabled = true;
                LuHy_VFTextBox.Enabled = true;
                Cpf_LenMetersTextBox.Enabled = true;
                Cablek1FreqTransfKTextBox.Enabled = true;
                CableZimagTextBox.Enabled = true;
                StubTermRTextBox.Enabled = true;
                StubTermLTextBox.Enabled = true;
                StubTermCTextBox.Enabled = true;
                Cablek2sqrtFreqTextBox.Enabled = true;
                CablecomboBox.Enabled = true;
                ValFreqGroupBox.Enabled = false;
                ParamValueGroupBox.Enabled = true;
                ForcegroupBox.Enabled = false;
                ForcecheckBox.Checked = false;
            }

            if (seriesZseriesbutton.Checked || seriesZparallelbutton.Checked)
            {
                e.Graphics.DrawImage(tilebitmaps.Series_Z, rect);
                R_Zreal_TrRatioLabel.Text = "Resistance, Ohms";
                LuHy_VFLabel.Text = "Inductance, uHy";
                Cpf_LenMetersLabel.Text = "Capacitance, pF";
                Cablek2sqrtFreqLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableZimagLabel.Text = "";
                StubTermRLabel.Text = "";
                StubTermLLabel.Text = "";
                StubTermCLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "";
                CableTypelabel.Text = "";
                R_Zreal_TrRatioTextBox.Enabled = true;
                LuHy_VFTextBox.Enabled = true;
                Cpf_LenMetersTextBox.Enabled = true;
                Cablek1FreqTransfKTextBox.Enabled = false;
                CableZimagTextBox.Enabled = false;
                StubTermRTextBox.Enabled = false;
                StubTermLTextBox.Enabled = false;
                StubTermCTextBox.Enabled = false;
                Cablek2sqrtFreqTextBox.Enabled = false;
                CablecomboBox.Enabled = false;
                ValFreqGroupBox.Enabled = false;
                ParamValueGroupBox.Enabled = true;
                ForcegroupBox.Enabled = false;
                ForcecheckBox.Checked = false;
            }

            if (transformerbutton.Checked)
            {
                e.Graphics.DrawImage(tilebitmaps.Transformer, rect);
                R_Zreal_TrRatioLabel.Text = "Turns ratio: SecondaryTurns/PrimaryTurns";
                LuHy_VFLabel.Text = "Primary winding inductance, uHy";
                Cpf_LenMetersLabel.Text = "";
                Cablek2sqrtFreqLabel.Text = "";
                Cablek1FreqTransfKLabel.Text = "Coupling coefficient: -1.0 to 0 to +1.0 (negative -> polarity reversal)";
                CableZimagLabel.Text = "";
                StubTermRLabel.Text = "";
                StubTermLLabel.Text = "";
                StubTermCLabel.Text = "";
                CableTypelabel.Text = "";
                R_Zreal_TrRatioTextBox.Enabled = true;
                LuHy_VFTextBox.Enabled = true;
                Cpf_LenMetersTextBox.Enabled = false;
                Cablek1FreqTransfKTextBox.Enabled = true;
                CableZimagTextBox.Enabled = false;
                StubTermRTextBox.Enabled = false;
                StubTermLTextBox.Enabled = false;
                StubTermCTextBox.Enabled = false;
                Cablek2sqrtFreqTextBox.Enabled = false;
                CablecomboBox.Enabled = false;
                ValFreqGroupBox.Enabled = false;
                ParamValueGroupBox.Enabled = true;
                ForcegroupBox.Enabled = false;
                ForcecheckBox.Checked = false;
            }

            if (pibutton.Checked)   // pibutton is disabled,
            {                       // so this block can't be executed
                e.Graphics.DrawImage(tilebitmaps.Pi_Net, rect);
                //R_Zreal_TrRatioLabel.Text = "TODO";
                //LuHy_VFLabel.Text = "TODO";
                //Cpf_LenMetersLabel.Text = "TODO";
            }

            if (teebutton.Checked)  // teebutton is disabled,
            {                       // so this block can't be executed
                e.Graphics.DrawImage(tilebitmaps.Tee_Net, rect);
                //R_Zreal_TrRatioLabel.Text = "TODO";
                //LuHy_VFLabel.Text = "TODO";
                //Cpf_LenMetersLabel.Text = "TODO";
            }
        }


        /// <summary>
        /// Occurs when the index of the selected cable type changes
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void CableTypeChanged(object sender, EventArgs e)
        {
            // Get new cable type that was selected
            cable = new CableSet((CableType)CablecomboBox.SelectedIndex);


            // fill in the table values
            LuHy_VFTextBox.Text = cable.Vf.ToString();
            Cablek1FreqTransfKTextBox.Text = cable.K1.ToString();
            Cablek2sqrtFreqTextBox.Text = cable.K2.ToString();
            R_Zreal_TrRatioTextBox.Text = cable.Zchar.real.ToString();
            CableZimagTextBox.Text = cable.Zchar.imag.ToString();

            // repaint the screen
            Refresh();
        }
    }
}
