/*******************************************************************************
* File Name: VDACI.c  
* Version 1.50
*
*  Description:
*    This file provides the source code to the API for the 8-bit Voltage DAC 
*    (VDAC8) User Module.
*
*   Note:
*     Any unusual or non-standard behavior should be noted here. Other-
*     wise, this section should remain blank.
*
*******************************************************************************
* Copyright 2008-2010, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions, 
* disclaimers, and limitations in the end user license agreement accompanying 
* the software package with which this file was provided.
********************************************************************************/

#include "cytypes.h"
#include "VDACI.h"

uint8 VDACI_initVar = 0;


/*******************************************************************************
* Function Name: VDACI_Init
********************************************************************************
* Summary:
*     Initialize to the schematic state
* 
* Parameters:  
*  void:  
*
* Return: 
*  (void)
*
* Theory: 
*
* Side Effects: 
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACI_Init(void) 
{
    VDACI_CR0 = (VDACI_MODE_V )  ;    

    /* Set default data source */
    if(VDACI_DEFAULT_DATA_SRC != 0 )    
    {
        VDACI_CR1 = (VDACI_DEFAULT_CNTL | VDACI_DACBUS_ENABLE) ;   
    }
    else
    {
        VDACI_CR1 = (VDACI_DEFAULT_CNTL | VDACI_DACBUS_DISABLE) ;  
    } 
    
    /* Set default strobe mode */
    if(VDACI_DEFAULT_STRB != 0)
    {
        VDACI_Strobe |= VDACI_STRB_EN ;
    }
    
    /* Set default range */
    VDACI_SetRange(VDACI_DEFAULT_RANGE); 
    
    /* Set default speed */
    VDACI_SetSpeed(VDACI_DEFAULT_SPEED);
}


/*******************************************************************************
* Function Name: VDACI_Enable
********************************************************************************
* Summary:
*     Enable the VDAC8
* 
* Parameters:  
*  void:  
*
* Return: 
*  (void)
*
* Theory: 
*
* Side Effects: 
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACI_Enable(void) 
{
    VDACI_PWRMGR |= VDACI_ACT_PWR_EN;
    VDACI_STBY_PWRMGR |= VDACI_STBY_PWR_EN;
}


/*******************************************************************************
* Function Name: VDACI_Start
********************************************************************************
*
* Summary:
*  The start function initializes the voltage DAC with the default values, 
*  and sets the power to the given level.  A power level of 0, is the same as executing
*  the stop function.
*
* Parameters:  
*  Power:   Sets power level between off (0) and (3) high power
*
* Return: 
*  (void) 
*
* Global variables:
*  VDACI_initVar: Is modified when this function is called for the first 
*   time. Is used to ensure that initialization happens only once.
*
* Reenrancy: 
*  No
*
*******************************************************************************/
void VDACI_Start(void) 
{  
    /* Hardware initiazation only needs to occure the first time */
    if ( VDACI_initVar == 0)  
    { 
        VDACI_Init();
       
        VDACI_initVar = 1;
    }  
   
    /* Enable power to DAC */
    VDACI_Enable();
    
    /* Set default value */
    VDACI_SetValue(VDACI_DEFAULT_DATA); 
}


/*******************************************************************************
* Function Name: VDACI_Stop
********************************************************************************
*
* Summary:
*  Powers down DAC to lowest power state.
*
* Parameters:  
*  (void)  
*
* Return: 
*  (void) 
*
* Theory: 
*
* Side Effects:
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACI_Stop(void) 
{
   /* Disble power to DAC */
   VDACI_PWRMGR &= ~VDACI_ACT_PWR_EN;
   VDACI_STBY_PWRMGR &= ~VDACI_STBY_PWR_EN;
}


/*******************************************************************************
* Function Name: VDACI_SetSpeed
********************************************************************************
*
* Summary:
*  Set DAC speed
*
* Parameters:  
*  power:   Sets speed value 
*
* Return: 
*  (void) 
*
* Theory: 
*
* Side Effects:
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACI_SetSpeed(uint8 speed) 
{
   /* Clear power mask then write in new value */
   VDACI_CR0 &= ~VDACI_HS_MASK ;    
   VDACI_CR0 |=  ( speed & VDACI_HS_MASK) ;    
}


/*******************************************************************************
* Function Name: VDACI_SetRange
********************************************************************************
*
* Summary:
*  Set one of three current ranges.
*
* Parameters:  
*  Range:  Sets one of Three valid ranges.
*
* Return: 
*  (void) 
*
* Theory: 
*
* Side Effects:
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACI_SetRange(uint8 range) 
{
   VDACI_CR0 &= ~VDACI_RANGE_MASK ;      /* Clear existing mode */
   VDACI_CR0 |= ( range & VDACI_RANGE_MASK ) ; /*  Set Range  */
   VDACI_DacTrim();
}


/*******************************************************************************
* Function Name: VDACI_SetValue
********************************************************************************
*
* Summary:
*  Set 8-bit DAC value
*
* Parameters:  
*  value:  Sets DAC value between 0 and 255.
*
* Return: 
*  (void) 
*
* Theory: 
*
* Side Effects:
*
* Reenrancy: 
*  Yes
*******************************************************************************/
void VDACI_SetValue(uint8 value) 
{
   VDACI_Data = value;                             /*  Set Value  */
   
   /* PSOC3 silicons earlier to ES3 require a double write */
   #if (VDACI_PSOC3_ES2 || VDACI_PSOC5_ES1)
       VDACI_Data = value;
   #endif
}


/*******************************************************************************
* Function Name: VDACI_DacTrim
********************************************************************************
*
* Summary:
*  Set the trim value for the given range.
*
* Parameters:  
*   range:  1V or 4V range.  See constants.  
*
* Return: 
*  (void) 
*
* Theory: 
*
* Side Effects:
*
* Reenrancy: 
*  Yes
*
*******************************************************************************/
void VDACI_DacTrim(void) 
{
    uint8 mode;
    
    mode = ((VDACI_CR0 & VDACI_RANGE_MASK) >> 2) + VDACI_TRIM_M7_1V_RNG_OFFSET;
    
    VDACI_TR = CY_GET_XTND_REG8((uint8 *)(VDACI_DAC_TRIM_BASE + mode));
}


/* [] END OF FILE */
