;*************************************************************************
; CWID.ASM - Repeater CW IDer based on the PIC 16C54
; By Gary C. Sutcliffe, W9XT
; Copyright 1995,  Gary C. Sutcliffe
; Rev 1.01    6 OCT 95
;*************************************************************************
; PIC Register and status bit definitions
;
status  equ     3h              ; F3 Reg is STATUS Reg.
f       equ     1               ;destination codes
w       equ     0

CARRY   equ     0h              ; Carry Bit is Bit.0 of F3
C       equ     0h
DCARRY  equ     1h
DC      equ     1h
Z_bit   equ     2h              ; Bit 2 of F3 is Zero Bit
Z       equ     2h
P_DOWN  equ     3h
PD      equ     3h
T_OUT   equ     4h
TO      equ     4h
PC      equ     2h      ;program counter
porta   equ     5h      ;16C5X Port A
portb   equ     6h      ;16C5X Port B
portc   equ     7h      ;16C5X Port C - not implemented in 16C54, mem only
;********************************************************************

;********************************************************************
; Pin and variable definitions, program constants

ptt      equ     0h      ;Port B bit 0 (pin 10) PTT output
cwtone   equ     1h      ;Port B bit 1 (pin 11) morse tone output
cwkey    equ     2h      ;Port B bit 2  (pin 12) keyed morse output
sqin     equ     0h      ;Port A bit 0 (pin 6) indicates rcvr squelch broken

TFREQ    equ     0b7h    ;constant to ctl tone freq, hi value = low freq
                         ;b7h gives ~900 Hz with 4MHz xtal
BITTIME  equ     036h    ;BITTIME determines how many cycles of the cw tone
                         ;frequency are required to equal the period of one
                         ; "dit" at the cw speed.  At 20 WPM, a bit time =
                         ; ~60 msec.  900 Hz => 1.11 msec.  60/1.11 = 54
                         ;54 => 36h.
flags    equ     08h     ;flags for program control
sqflg    equ     00h     ;squelch tripped flag in flags byte
wait     equ     01h     ;wait flag in flags
tmp      equ     09h     ; GP temp variables
tmp2     equ     0ah
lcnt     equ     0bh     ;loop count variable
cidx     equ     0ch     ;index into call sign table
bitcntr  equ     0dh     ;bit counter, bit position in current byte
cwbyte   equ     0eh     ;byte to parse for sending CW
mscntr   equ     0fh     ;counts 50 millisecond delays
tensec  equ     10h     ;10 second  interval counter
;**********************************************************************
reset  org 01ffh
       goto start


       org 0
start  clrw              ;set up option reg
       option
       call setup        ; setup i/o pin configs
       bcf portb, cwkey  ;be sure key is up at start
       call id           ;start out with an ID at startup or reset
       bcf flags,wait     ;don't wait 10 minutes to ID
       bcf flags,sqflg    ;start with no squelch broken
       call rsttime       ;reset 10 minute timer regs

; This is the main loop that handles when to ID

main
       btfss porta,sqin  ;check if rcvr squelch is broken (ie xmiting)
       bsf flags,sqflg   ;set the flag if it is
       btfss flags,sqflg ;see if squelch flag set
       goto wtsome       ;if not, take some time off timer
       btfsc flags,wait  ;if we still have to wait, wait some more
       goto wtsome
       call id           ;transmitted & have not IDed in 10 min, ID now
       call wait2min     ;After ID, wait 2 min before checking xmit status
       bsf flags,wait    ;reset wait flag
       bcf flags,sqflg   ;clear squelch flag
       call rsttime      ;reset 10 minute timers
wtsome call del50ms      ;kill 50 msec
       decfsz mscntr     ;check if msec counter = 0
       goto main         ;not this pass, try again
       movlw 0c8h        ;init 50 msec counter
       movwf mscntr      ;
       decfsz tensec     ;check if 10sec timer expired
       goto main         ;nope, go try again
       bcf flags,wait    ; tell them we don't have to wait to ID
       goto main

;**************************************************************************
;ID - sends call sign and operates PTT

id
       bsf portb,ptt     ;hit the PTT
       clrf cidx         ;set index to first byte in table
nxtbyt movf cidx,w       ;get index in W for calc table offset after call
       call calltbl      ;get next byte from table
       movwf cwbyte      ;store byte
       movf cwbyte,f     ;see if its = 0
       btfsc status,Z    ;
       goto calldn       ;if =0, done sending call
       movlw 8
       movwf bitcntr     ;init bitcounter
nxtbit btfss cwbyte,7    ;check most significant bit
       goto do0
       call keydown       ;hold key down 1 dit period
       goto newdit
do0    call keyup        ;key up 1 dit period
newdit rlf cwbyte,f      ;rotate next bit into position
       decfsz bitcntr,f  ;see if we did all bits in this byte
       goto nxtbit       ;check the next bit in the byte of CW
       incf cidx         ;inc index to next byte in call sign table
       goto nxtbyt       ;get the next byte from call sign table

calldn bcf portb, cwkey        ;be sure we end with key up
       bcf portb, ptt          ;release the PTT
       retlw 0                 ;end of ID

;************************************************
; SETUP - configures I/O pins

setup   clrw
        tris portb       ;set port B as all outputs
        movlw 0fh        ; make port A all inputs
        tris porta       ; set port A
        retlw  1         ;return to main prog

;*********************************************************
; KEYDOWN - This sends a dit.  Note that this routine returns with
;cwkey in true (key down) position.

keydown bsf portb, cwkey  ;key down
        movlw BITTIME     ;go through loop this many times
        movwf lcnt
onecycd bsf portb,cwtone  ;set bit hi for first 1/2 of cycle
        movlw TFREQ       ;TFREQ is number of times through loop which
        movwf tmp         ;determines the tone freq
dlhi    decfsz tmp,f      ;decrement until 0
        goto dlhi
        bcf portb,cwtone  ;now clear the output pin for 2nd 1/2
        movlw TFREQ
        movwf tmp
dllow   decfsz tmp,f      ;wait until 0 to finish cycle
        goto dllow
        decfsz lcnt,f     ;
        goto onecycd      ;go do another cycle
        retlw 0           ;


;*********************************************************
; KEYUP - This is a delay for one bit period.  Note that this routine
; returns with cwkey in false (key up) position.  This routine is essentially
; the same as KEYDOWN except it does not toggle cwtone.  Instead it toggles
; a dummy bit in the tmp2 variable.  This is to keep the timing of the two
; routines the same.

keyup   bcf portb, cwkey  ;key up
        movlw BITTIME     ;go through loop this many times
        movwf lcnt
onecycu bsf tmp2,cwtone   ;simulate set bit hi for first 1/2 of cycle
        movlw TFREQ       ;TFREQ is number of times through loop which
        movwf tmp         ;determines the tone freq
ulhi    decfsz tmp,f      ;decrement until 0
        goto ulhi
        bcf tmp2,cwtone  ;simulate clearing the output pin for 2nd 1/2
        movlw TFREQ
        movwf tmp
ullow   decfsz tmp,f      ;wait until 0 to finish cycle
        goto ullow
        decfsz lcnt,f     ;
        goto onecycu      ;go do another cycle
        retlw 0           ;

;*****************************************************************
; RSTTIME  - reset the registers that handle the 10 minute timer
; Because we wait 2 minutes after each ID, this is set up for 8 minutes
rsttime movlw 0c8h        ;init 50 msec counter
        movwf mscntr      ;when decr down to 0, 10 seconds elapsed
        movlw 30h        ;init 10 second counter for 8 min
        movwf tensec     ;10 min elapsed when = 0
        retlw 0



;**********************************************************************
; DEL50MS - software delay for 50 milliseconds.  Based on 4 MHz clock
;           Uses nested loop to generate delay

del50ms
        movlw 27h    ;Outer loop value
        movwf tmp
dl50o   movlw 0ffh    ;initialize inner loop value
        movwf tmp2
dl50i   goto dlnxt    ;fancy way of killing 2 cycles with 1 instruction
dlnxt   decfsz tmp2
        goto dl50i    ;end of inner loop
        decfsz tmp    ;check if outer loop done
        goto dl50o    ;if not restart inner loop
        retlw 0       ;return when done

;************************************************************
;WAIT2MIN - Wait for 2 minutes.  After IDing, we will wait 2 minutes before
;checking if we will ID again because the repeater has been brought up again.
;This is to discourage kerchunkers and to prevent an endless ID every 10 min
;due to the repeater staying on a few seconds after IDing, and detecting
;the squelch being broken.

wait2min
        movlw 0c8h        ;init 50 msec counter
        movwf mscntr      ;when decr down to 0, 10 seconds elapsed
        movlw 0ch        ;init 10 second counter  for 2 min
        movwf tensec     ;2 min elapsed when = 0
wt2lp  call del50ms      ;kill 50 msec
       decfsz mscntr     ;check if msec counter = 0
       goto wt2lp        ;not this pass, try again
       movlw 0c8h        ;init 50 msec counter to count to 10 sec
       movwf mscntr      ;
       decfsz tensec     ;check if 10sec timer expired
       goto wt2lp        ;nope, go try again
       retlw 0           ;2 minutes are up, bug out


;************************************************************
; CALLTBL - Callsign table.  This is were the call sign is put.  A 0
; is key up, 1 = keydown.  A dit is a single 1, a dah is three 1's
; in a row.  Put a 0 between dits & dahs in a letter. Put 3 0's between
; letters.  Put a few 0's at the very beggining so that the PTT relay
; has time to close before the CW starts.  The 1's & 0's are examined
; from left to right.
; The very last line must be all zeros.  That is what indicates the
; end of the ID string.

calltbl addwf PC,f       ;calc offset into table for next byte of CW
                         ;This table holds "DE N9LZW/R"
        retlw  B'00000001'   ;D..
        retlw  B'11010100'   ;..D
        retlw  B'01000001'   ;E N..
        retlw  B'11010001'   ;..N9..
        retlw  B'11011101'   ;..9..
        retlw  B'11011101'   ;..9
        retlw  B'00010111'   ;L..
        retlw  B'01010001'   ;..LZ..
        retlw  B'11011101'   ;..Z
        retlw  B'01000101'   ;Z..W..
        retlw  B'11011100'   ;..W..
        retlw  B'01110101'   ;../..
        retlw  B'01110100'   ;../..
        retlw  B'01011101'   ;..R
        retlw  B'00000000'   ; all zeros indicate end of table

        end               ;end of CWID
