from myhdl import *
import random, struct, string
from divideBy10 import divideBy10
from config import *

def insertChar(clk, start, k, char, result):
	# Verilog requires fixed range values.
	# MyHDL does not
	@always(clk.negedge)
	def logic():
		if start == ACTIVE_LOW:
			if k == 0:
				result.next[8:0] = char
			elif k == 1:
				result.next[16:8] = char
			elif k == 2:
				result.next[24:16] = char
			elif k == 3:
				result.next[32:24] = char
			elif k == 4:
				result.next[40:32] = char
			elif k == 5:
				result.next[48:40] = char
			elif k == 6:
				result.next[56:48] = char
			elif k == 7:
				result.next[64:56] = char
			elif k == 8:
				result.next[72:64] = char
			elif k == 9:
				result.next[80:72] = char
			else:
				pass
	return logic

def lcdFormatP(clk, reset, fo_start, fo_done, sigValue, fo_result,
				   ud_start, ud_done, udividend, uquotient, initialDividend,
				   ic_start, ic_k, ic_char):
	"""
		Format counter value

		Converts a binary value (0 to 99,999,999) to a string of hex ascii digits.
		This is done by repeated divisions by 10. The final value has
		has commas inserted in the appropriate places. The binary value 2345678
		would return 20322c3334352c363738.

		Inputs:
			clk
			fo_start
			sigValue	-	Binary value to be converted to ascii characters
							This is also the dividend argument for divideBy10
		Outputs:
			fo_done
			fo_result	-	Array containing the formatted string [80:]
							Will contain 10 ascii characters
							including leading blanks.
	"""

	State = enum('STATE1', 'STATE2', 'STATE3', 'STATE4', 'STATE4A', 'STATE5',
				 'STATE6', 'STATE7', 'STATE8', 'STATE9', 'STATE10', 'DEFAULT')
	state = Signal(State.DEFAULT)

	@always(clk.negedge, reset.negedge)
	def logic():
		if reset == ACTIVE_LOW:
			state.next = State.DEFAULT
		elif fo_start == ACTIVE_LOW:
			state.next = State.STATE1
			fo_done.next = INACTIVE_HIGH
			ic_k.next = 0
		else:
			if state == State.STATE1:
				initialDividend.next = sigValue
				ud_start.next = ACTIVE_LOW		# start udivide for the first digit
				state.next = State.STATE2

			elif state == State.STATE2:			# top of digit processing loop
				ud_start.next = INACTIVE_HIGH
				state.next = State.STATE3

			elif state == State.STATE3:
				if ud_done == ACTIVE_LOW:		# wait for udivide to complete
					state.next = State.STATE4

			elif state == State.STATE4:
				if ic_k == 3 or ic_k == 7:			# is a comma required here?
					ic_char.next = 0x2c
					ic_start.next = ACTIVE_LOW		# start insertChar
					state.next = State.STATE4A
				else:
					state.next = State.STATE6

			elif state == State.STATE4A:
				ic_start.next = INACTIVE_HIGH
				state.next = State.STATE5

			elif state == State.STATE5:
				ic_k.next = ic_k + 1
				state.next = State.STATE6

			elif state == State.STATE6:
				ic_char.next = udividend[4:]+0x30	# insert the next digit
				ic_start.next = ACTIVE_LOW			# start insertChar
				state.next = State.STATE7

			elif state == State.STATE7:
				ic_start.next = INACTIVE_HIGH
				state.next = State.STATE8

			elif state == State.STATE8:
				if uquotient == 0:
					state.next = State.STATE9	# done with the digits
				else:
					ic_k.next = ic_k + 1
					initialDividend.next = uquotient
					ud_start.next = ACTIVE_LOW	# Start udivide for remaining digits
					state.next = State.STATE2

			elif state == State.STATE9:
				if ic_k < 10:					# insert leading blanks
					ic_k.next = ic_k + 1
					ic_char.next = 0x20
					ic_start.next = ACTIVE_LOW	# start insertChar
					state.next = State.STATE10
				else:
					fo_done.next = ACTIVE_LOW
					state.next = State.DEFAULT

			elif state == State.STATE10:
				ic_start.next = INACTIVE_HIGH
				state.next = State.STATE9

			elif state == State.DEFAULT:
				pass
			else:
				state.next = State.DEFAULT

	return logic

def lcdFormat(clk, reset, fo_start, fo_done, sigValue, fo_result):

	ic_k = Signal(intbv(0)[7:])
	ic_char = Signal(intbv(0)[8:])
	ic_start = Signal(bool(INACTIVE_HIGH))
	ud_start, ud_done = [Signal(bool(INACTIVE_HIGH)) for k in range(2)]
	udividend, uquotient, initialDividend = [Signal(intbv(0)[COUNT_WIDTH:]) for k in range(3)]

	DIV = divideBy10(clk, reset, ud_start, ud_done, udividend, uquotient, initialDividend)
	IC = insertChar(clk, ic_start, ic_k, ic_char, fo_result)
	FO = lcdFormatP(clk, reset, fo_start, fo_done, sigValue, fo_result,
				   ud_start, ud_done, udividend, uquotient, initialDividend,
				   ic_start, ic_k, ic_char)
	return instances()

#=====================================================================================
# test code follows

def fixup(s):
	""" Insert commas into a string of decimal digits. """
	n = len(s)
	if n < 4:
		return s
	elif n == 4:
		s = s[:1] + ',' + s[1:]
	elif n == 5:
		s = s[:2] + ',' + s[2:]
	elif n == 6:
		s = s[:3] + ',' + s[3:]
	elif n == 7:
		s = s[:1] + ',' + s[1:4] + ',' + s[4:]
	else:
		s = s[:2] + ',' + s[2:5] + ',' + s[5:]
	return s

def formatResults(s):
	""" Convert the string of binary hex ascii values to a string
		eg: 35372c3634362c303734 -> 57,646,074
	"""
	fs = ''
	for k in range(0, len(s), 2):
		x = s[k:k+2]
		y = int(string.atoi(x, 16))
		fs += chr(y)
	fs = fs.strip()
	return fs


def testBench():

	clk, fo_start, fo_done = [Signal(bool(INACTIVE_HIGH)) for k in range(3)]
	counterVal = Signal(intbv(0)[32:])
	result = Signal(intbv(0)[80:])
	reset = Signal(bool(ACTIVE_LOW))

	formatInst = lcdFormat(clk, reset, fo_start, fo_done, counterVal, result)


	HALF_PERIOD = delay(1)
	@always(HALF_PERIOD)
	def clockGen():
		clk.next = not clk

	@instance
	def testLcdFormat():
		yield clk.posedge
		reset.next = INACTIVE_HIGH
		print 'now', now()

		N = 300
		maxv = 0
		minv = 0xffffffff
		vals =(40000000, 0, 1, 12, 123, 4000, 12345, 123456, 1234567, 99965432)
		vals = (123, 12345678)
		#for k in range(2):
		for v in vals:
			#v = random.randint(0, 99999999)
			maxv = max(v, maxv)
			minv = min(v, minv)
			print 'v', v

			counterVal.next = v
			fo_result = 0
			fo_start.next = ACTIVE_LOW
			yield clk.negedge
			fo_start.next = INACTIVE_HIGH		# start lcdFormat
			yield fo_done.negedge
			s = hex(result)[2:-1]		# remove the 0x at the beginning and the L at the end.
			print 'result', hex(result), len(result)
			#print 's', s
			fs = formatResults(s)
			v = str(v)
			vs = fixup(v)
			print 'expected', vs
			print 'actual    ', fs
			if fs != vs:
				print 'Error:', fs, '  ', vs
			print
		print 'min:', minv, '  max:', maxv
		print 'done', now()
		raise StopSimulation


	return instances()

def test3():
	clk, fo_start, fo_done = [Signal(bool(INACTIVE_HIGH)) for k in range(3)]
	counterVal = Signal(intbv(0)[32:])
	result = Signal(intbv(0)[80:])
	reset = Signal(bool(ACTIVE_LOW))
	toVerilog(lcdFormat, clk, reset, fo_start, fo_done, counterVal, result)

def test1():
	tb = testBench()
	Simulation(tb).run(5000)

if __name__ == '__main__':
	test3()