from myhdl import *
from math import log, ceil
from config import *

DELAY100US = int(MASTER_CLOCK_FREQUENCY * 100e-6)
DELAY1MS  = int(MASTER_CLOCK_FREQUENCY * 1e-3)
DELAY2MS  = int(MASTER_CLOCK_FREQUENCY * 2e-3)
DELAY5MS  = int(MASTER_CLOCK_FREQUENCY * 5e-3)
DELAY16MS = int(MASTER_CLOCK_FREQUENCY * 16e-3)
DELAY20MS = int(MASTER_CLOCK_FREQUENCY * 20e-3)
DELAY_WIDTH = int(ceil(log(DELAY20MS, 2)))

if debug:
	WRITE_DELAY_COUNT = 5	# test only
	WRITE_SETUP_COUNT = 1
	DELAY_WIDTH += 1
else:
	WRITE_DELAY_COUNT = int(100e-6 * MASTER_CLOCK_FREQUENCY)
	WRITE_SETUP_COUNT = int(50e-6 * MASTER_CLOCK_FREQUENCY)


def lcdController(clk, reset, start, done, mode, dbyte, c1, c2, lcdBus, E, RS, RW):
	"""
	Initializes and performs I/O for a
	Optrex LCD display operated in the 4-bit mode.

	Inputs:
		clk
		reset
		start
		mode	-	0 initialize display, 1 issue command, 2 write data byte
		dbyte	-	Command or data byte
						commands
							1	-	clear display
							2	- 	home cursor
							3	-	position cursor
									cmd = 0x80 | ((row & 1)<<6) | (col & 0xf)
													row: 0 - 1;  col: 0 - 15
		c1, c2	-	Command data bytes

	Outputs:
		done
		lcdBus	-	4 bits
		E		-	Data strobe
		RS		-	Register select: 0 command, 1 data
		RW		-	Read/write select: 0 write, 1 read
	"""
	State = enum('STATE0','STATE1', 'STATE2', 'STATE2A', 'STATE3', 'STATE4',
				 'STATE5', 'STATE6', 'STATE7', 'IDLE', 'DELAY',
				 'WRITE1', 'WRITE2', 'WRITE3', 'WRITE4')
	state, returnState, writeReturnState = [Signal(State.IDLE) for k in range(3)]
	delayCount = Signal(intbv(0)[DELAY_WIDTH:])
	commands = tuple([0xc, 6, 1, 8, 0x28, 2, 3, 3, 3])	# Initialization commands in reverse order
	numNibs = tuple([   1, 1, 1, 1,    1, 0, 0, 0, 0])	# 0-write one nibble, 1-write two nibbles.
	nibs = Signal(intbv(0)[1:])
	delays = tuple([DELAY2MS, DELAY2MS, DELAY16MS, DELAY2MS,
				    DELAY2MS, DELAY1MS, DELAY1MS, DELAY1MS, DELAY5MS])
	k = Signal(intbv(0)[4:])
	wbyte = Signal(intbv(0)[8:])

	@always(clk.negedge, reset.negedge)
	def logic():
		if reset == ACTIVE_LOW:
			state.next = State.IDLE
			done.next = INACTIVE_HIGH
		elif start == ACTIVE_LOW:
			RW.next = 0			# write
			done.next = INACTIVE_HIGH
			state.next = State.STATE0
		else:
			if state == State.STATE0:
				if mode == 0:
					# Begin display initialization
					RS.next = 0			# select command register
					k.next = 8			# loop index
					delayCount.next = DELAY16MS;	# power-on delay
					returnState.next = State.STATE2
					state.next = State.DELAY
				elif mode == 1:
					# issue command
					RS.next = 0
					if dbyte == 3:
						# position cursor:                  row                         col
						wbyte.next = 0x80 | ((c1 & 1)<<6) | (c2 & 0xf)
					else:
						wbyte.next = dbyte
					returnState.next = State.STATE1
					delayCount.next = WRITE_SETUP_COUNT
					state.next = State.DELAY
				else:
					# write data byte
					RS.next = 1
					wbyte.next = dbyte
					returnState.next = State.STATE1
					delayCount.next = WRITE_SETUP_COUNT		#!!! this could be skipped if RS is already set.
					state.next = State.DELAY				#    but hardy necessary.

			elif state == State.STATE1:
				if mode == 1:
					writeReturnState.next = State.STATE5	# command
				else:
					writeReturnState.next = State.STATE7	# data byte
				state.next = State.WRITE1

			# states 2 - 4: initialization
			elif state == State.STATE2:
				wbyte.next = commands[k]
				nibs.next = numNibs[k]
				state.next = State.STATE2A

			elif state == State.STATE2A:
				if nibs == 0:
					state.next = State.WRITE3		# write one nibble
				else:
					state.next = State.WRITE1		# write two nibbles
				writeReturnState.next = State.STATE3

			elif state == State.STATE3:
				delayCount.next = delays[k]
				returnState.next = State.STATE4
				state.next = State.DELAY

			elif state == State.STATE4:
				if k == 0:
					done.next = ACTIVE_LOW
					state.next = State.IDLE
				else:
					k.next = k - 1
					state.next = State.STATE2

			# states 5 - 6: conclusion of write command
			elif state == State.STATE5:
				if dbyte == 1 or dbyte == 2:	# clear display / home cursor
					delayCount.next = DELAY20MS
				else:
					delayCount.next = DELAY100US
				returnState.next = State.STATE6
				state.next = State.DELAY

			elif state == State.STATE6:
				done.next = ACTIVE_LOW
				state.next = State.IDLE

			# conclusion of write data byte
			elif state == State.STATE7:
				delayCount.next = DELAY100US
				returnState.next = State.STATE6
				state.next = State.DELAY

			elif state == State.DELAY:
				# The delay is <delayCount> + 2 clocks
				if delayCount == 0:
					state.next = returnState
				else:
					delayCount.next = delayCount - 1

			elif state == State.WRITE1:
				lcdBus.next = wbyte[8:4]
				E.next = 1
				delayCount.next = WRITE_DELAY_COUNT
				returnState.next = State.WRITE2
				state.next = State.DELAY

			elif state == State.WRITE2:
				E.next = 0
				delayCount.next = WRITE_DELAY_COUNT
				returnState.next = State.WRITE3
				state.next = State.DELAY

			elif state == State.WRITE3:
				lcdBus.next = wbyte[4:]
				E.next = 1
				delayCount.next = WRITE_DELAY_COUNT
				returnState.next = State.WRITE4
				state.next = State.DELAY

			elif state == State.WRITE4:
				E.next = 0
				delayCount.next = WRITE_DELAY_COUNT
				returnState.next = writeReturnState
				state.next = State.DELAY

			elif state == State.IDLE:
				pass
			else:
				state.next = State.IDLE

	return instances()

def test1():
	clk, reset, start, done, E, RS, RW = [Signal(bool(0)) for k in range(7)]
	mode = Signal(intbv(0)[2:])
	dbyte, c1, c2 = [Signal(intbv(0)[8:]) for k in range(3)]
	lcdBus = Signal(intbv(0)[4:])

	toVerilog(lcdController, clk, reset, start, done, mode, dbyte, c1, c2, lcdBus, E, RS, RW)

if __name__ == '__main__':
	test1()
