#!/usr/bin/python3
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#
# This software uses the PYTTSX3 voice synthesis package to generate spoken English from various
# inputs - 32 sentences in a stored data file that are enabled with Python radio buttons, as well
# as a "live" text input buffer and a "phonetic" text buffer that will automatically convert text
# such as callsigns to the appropriate phonetic representation.
#
# The code here is written in Python 3 by John Wiseman, WT3J.
#
# It has been tested extensively with a Raspberry Pi 5 with 8GB of RAM with no issues.
# The software requires Python 3.11 or higher.
#
# Output is via 1 Raspberry Pi GPIO pin, used for both the radio-microphone PTT an LED monitor.
#
# The GUI is implemented with Tkinter graphics, allowing inputs via the keyboard or from
# STORED sentences selected with radio buttons.
#
# To make it easier and faster for other users to modify the radio buttons for their own
# callsigns and information, an external file import was added.  This external file is
# hardcoded below as SSB_Program_Data.py and must be saved as that title and in the same directory.
# Each radio button is defined with 2 consecutive changeable lines.  The first line is the
# character string that will appear in the GUI and the second line is the color of the text.
# While there is no real limit to the number of radio buttons that can be used, the maximum
# number is currently hardcoded at 32.  The way the program is structured allows for any number
# of radio buttons between 1 and 32 to be used in the GUI.  The GUI window is sized appropriately
# for 32, and will remain the same with less, although the right side child window will shrink
# vertically as the number used decreases.  Addition or subtraction of radio buttons must be
# done with both lines added or subtracted together, treated as pairs.  The number of radio
# buttons defined is determined by counting the number of lines in SSB_Program_Data.py and
# dividing by 2.

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Update this constant whenever the code is modified.
# This date will appear in the GUI window title box.
LAST_MODIFIED = "2/28/25"
#
# Added 100 ms delays to closing the audio stream and turning off the PTT to eliminate any speech
# cutoff noticed at the end of transmissions when monitoring on a separate receiver.  This had been
# mentioned during a QSO as well.

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Important note on generating a desktop program start icon -
# Raspberry Pi icon in system tray...
# Preferences...
# Main Menu Editor...
# Other...
# New Item...
# Command...
# /home/radiopi1/QST/venv/bin/python3 /home/radiopi1/QST/venv/WT3J_SSB_V1.py

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Library imports are done here.
# Import the time delay module.
import time

# Import regular expressions to use in the parsing of the incremental serial number to be used
# in the first radio button when incrementing or repeating contest serial numbers.
import re

# Import the Tkinter library for the GUI.
from tkinter import*
import tkinter.font #as font to have better control of the GUI text.

# Set up the GPIO pin needed in the Voice Interface Module and the radio PTT.
from gpiozero.pins.rpigpio import RPiGPIOFactory
from gpiozero import LED
factory = RPiGPIOFactory()
VOICE_SWITCH = LED(22, pin_factory=factory)

# Get the necessary support for streaming synthesized audio to the analog output.
import numpy as np
import sounddevice as sd
from piper.voice import PiperVoice

# IMPORTANT NOTE ON USING PIPER VOICE MODELS -
# Here is an example of how to test a Piper voice model, and at the same time download the required
# .onnx and .json files.
# echo 'Testing one two three.' | piper --model en_US-ryan-low --output_file ryan.wav
# DO THIS FOR EVERY VOICE MODEL TO BE USED.

# Allow for threading so the GUI buttons operate as planned while characters are transmitted.
# GUI radio buttons will appear 'live' during character transmission, but are 'locked out'
# to prevent the buffer from being corrupted.
# Any character transmission can be terminated with the 'Stop Transmitting' button at any
# time by using threading.
import threading

# SSB_Program_Data.py is where the radio button fixed text resides and is read from.
# The first line for each button is the actual text to display in the GUI.
# The second line is used for attributes, and it is currently used for text color.
# The code limits this file to 32 radio buttons. The format MUST be 2 lines for each button,
# with no additional lines, including comments.
import SSB_Program_Data

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Define a CONSTANT, GUI_WINDOW_SIZE, to be a size and placement that looks good on your monitor.
# These values are what I prefer on a 28" 4K monitor for the GUI window.
# The first 2 numbers are the horizontal and vertical sizes, respectively, for the GUI window.
# The last 2 numbers are the initial GUI window placement relative to the upper lefthand corner.
#GUI_WINDOW_SIZE = "1900x1310+50+700"
GUI_WINDOW_SIZE = "1900x1440+1800+580"


# Define a CONSTANT, RADIO_BUTTON_FONT, to be a font that looks good on your monitor.
# These values are what I prefer on a 28" 4K monitor for the GUI radio button text.
RADIO_BUTTON_FONT = "Arial Black", "16", "bold"
RADIO_BUTTON_FONT_DYNAMIC = "Arial Black", "16", "bold", "italic"

# Define a CONSTANT, LABEL_TEXT_FONT, to be a font that looks good on your monitor.
# These values are what I prefer on a 28" 4K monitor for the GUI label text.
LABEL_TEXT_FONT  = "FreeSans" ,"20", "bold", "italic"
LABEL_TEXT_FONT_SMALL  = "FreeSans" ,"12", "bold", "italic"

# Define a CONSTANT, ENABLE_BUTTON_FONT, to be a font that looks good on your monitor.
# These values are what I prefer on a 28" 4K monitor for the GUI enable button text.
ENABLE_BUTTON_FONT = "Arial Black", "16", "bold"

# Define a CONSTANT, ENTRY_BOX_TEXT_FONT, to be a font that looks good on your monitor.
# These values are what I prefer on a 28" 4K monitor for the GUI character entry box text.
ENTRY_BOX_TEXT_FONT = "Arial Black", "16", "bold"
SECONDARY_ENTRY_BOX_TEXT_FONT = "Arial Black", "16", "bold", "italic"

# Define a CONSTANT, ENTRY_BOX_CHARACTER_WIDTH, to be a size that looks good on your monitor.
# This value is what I prefer on a 28" 4K monitor for the GUI character entry box width.
ENTRY_BOX_CHARACTER_WIDTH = "50"

# Define a CONSTANT for the horizontal distance of the radio buttons from the left
# edge of the GUI window.
# This value can be adjusted for preference based on your monitor and other GUI sizings.
RADIO_BUTTON_HORIZONTAL_PAD = 35
VOICE_RADIO_BUTTON_HORIZONTAL_PAD_MODE = 220

# Define a CONSTANT, SLIDER_FONT, to be a font that looks good on your monitor.
# These values are what I prefer on a 28" 4K monitor for the GUI slider button text.
SLIDER_FONT = "Arial Black", "14", "bold"

# Define a CONSTANT to fit the sliders to the desired width.
# These values are what I prefer on a 28" 4K monitor for the GUI sliders.
SLIDER_WIDTH = 500

# Define a CONSTANT, LEFT_WINDOW_PAD_Y, to be used to align the bottom of the left window pane.
LEFT_WINDOW_PAD_Y = 35

# Define a CONSTANT, CENTER_WINDOW_PAD_Y, to be used to align the bottom of the center window pane.
CENTER_WINDOW_PAD_Y = 33

# Define a CONSTANT, RIGHT_WINDOW_PAD_Y, to be used to align the bottom of the right window pane.
RIGHT_WINDOW_PAD_Y = 9

#Define CONSTANTS for the LED monitor status radio buttons.
VOICE_SELECTION_1 = "US English Male Voice - 1"
VOICE_SELECTION_2 = "US English Female Voice - 1"
VOICE_SELECTION_3 = "US English Male Voice - 2"
VOICE_SELECTION_4 = "US English Female Voice - 2"

VOICE_RATE_SELECTION_1 = "Voice Rate - 2X Speed"
VOICE_RATE_SELECTION_2 = "Voice Rate - Normal"
VOICE_RATE_SELECTION_3 = "Voice Rate - 1/2 Speed"

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Define a CONSTANT, GUI_TITLE_TEXT, as the label text for the GUI window.
# The constant LAST_MODIFIED is appended to the label text as the date that the code
# was last modified.
GUI_TITLE_TEXT = "WT3J Voice Synthesizer " + LAST_MODIFIED

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Read the external file SSB_Program_Data.py from the same directory. This file will
# contain the text for each of the 32 radio buttons used for storing commonly used
# sentences to transmit. The number of lines in this file are counted and used later
# to allow any number of radio buttons to be dynamically used, 1 through 28.
radio_button_file_name = "/home/radiopi1/QST/venv/SSB_Program_Data.py"
line_count = 0
with open(radio_button_file_name, 'r') as input_file:
    for line in input_file:
        # Read the first character in each line.
        # If the line is a comment started with the character '#', then do NOT count it.
        first_character = input_file.read(1)
        if first_character != '#':
            line_count += 1
# The line count is divided by 2 here because there are currently 2 lines used for each button.
# The first one is the actual text, the second one is for attributes such as color.
radio_button_count = int(line_count/2)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Initialize the first radio button text to the first line in the data file.
# This may be changed by inputting text and clicking on 'Change Dynamic Text'.
# This global variable is ONLY modified in the tz_buffer_click() function after the button
# labeled "Change Dynamic Text" is clicked.  As such, side effects are not an issue.
dynamic_data_input = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_0

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Define the TKinter control window for all widgets.
# This coupled with the last line in the program - window.mainloop() - defines the window and
# allows it to continually run and wait for inputs to process.
window = Tk()
# Window title is named here.
window.title(GUI_TITLE_TEXT)

# Size the window appropriately for all widgets added.
# Place the sized window in a default position relative to the upper left-hand corner of the screen.
window.geometry(GUI_WINDOW_SIZE)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Define a CONSTANT for the window and radio button background color.
BACKGROUND_COLOR = "gray85"

# Let mainwindow be the "parent" window for the left and right side "children" windows.
mainwindow = Frame(window, relief=RAISED, bd = 10, bg = 'gray80')
mainwindow.pack(anchor = 'n')

# Define leftwindow to be a left side "child" window for the text widgets to reside in.
# Note that the term of "leftwindow" here is a hold-over from when there was another window
# to the left.
leftwindow = Frame(mainwindow, relief=RAISED, bd = 3, bg = BACKGROUND_COLOR)
leftwindow.pack(side = 'left', anchor = 'n', padx = 10, pady = 10)

# Define rightwindow to be a right side "child" window for the button widgets to reside in.
rightwindow = Frame(mainwindow, relief=RAISED, bd = 3, bg = BACKGROUND_COLOR)
rightwindow.pack(side = 'left', anchor = 'n', padx = 10, pady = 10)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Functions to transfer STORED sequence characters to transmit via radio buttons.
# Each radio button requires its own function to access a unique sequence.  Note that the first
# button is unique in that it can be updated dynamically, bypassing the data input file.
# 32 functions are given below, allowing for up to 32 radio buttons to be displayed.  Less than
# 32 may be defined in the data file and only that number will be displayed in the GUI.  But if
# more than 32 are desired, the number of functions here will need to be extended accordingaly.
def get_STORED_sequence_0():
    """Get the STORED character sequence for Radio Button 0 and pass to 'voice', the function that calls PYTTSX3.

    This radio button is unique from the others, as it may be updated dynamically with the text input box
    and the 'Change Dynamic Text' button.  This update will stay locked in until the program is reset.
    """
    if threading.active_count() <= 2:
        t1 = threading.Thread(target=voice, args=[dynamic_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_1():
    """Get the STORED character sequence for Radio Button 1 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_1
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_2():
    """Get the STORED character sequence for Radio Button 2 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_2
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_3():
    """Get the STORED character sequence for Radio Button 3 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_3
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_4():
    """Get the STORED character sequence for Radio Button 4 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_4
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_5():
    """Get the STORED character sequence for Radio Button 5 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_5
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_6():
    """Get the STORED character sequence for Radio Button 6 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_6
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_7():
    """Get the STORED character sequence for Radio Button 7 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_7
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_8():
    """Get the STORED character sequence for Radio Button 8 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_8
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_9():
    """Get the STORED character sequence for Radio Button 9 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_9
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_10():
    """Get the STORED character sequence for Radio Button 10 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_10
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_11():
    """Get the STORED character sequence for Radio Button 11 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_11
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_12():
    """Get the STORED character sequence for Radio Button 12 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_12
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_13():
    """Get the STORED character sequence for Radio Button 13 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_13
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_14():
    """Get the STORED character sequence for Radio Button 14 and pass to 'voice', the function that calls PYTTSX3."""
    t2 = threading.active_count()
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_14
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_15():
    """Get the STORED character sequence for Radio Button 15 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_15
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_16():
    """Get the STORED character sequence for Radio Button 16 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_16
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_17():
    """Get the STORED character sequence for Radio Button 17 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_17
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_18():
    """Get the STORED character sequence for Radio Button 18 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_18
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_19():
    """Get the STORED character sequence for Radio Button 19 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_19
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return
    
def get_STORED_sequence_20():
    """Get the STORED character sequence for Radio Button 20 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_20
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_21():
    """Get the STORED character sequence for Radio Button 21 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_21
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_22():
    """Get the STORED character sequence for Radio Button 22 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_22
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_23():
    """Get the STORED character sequence for Radio Button 23 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_23
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_24():
    """Get the STORED character sequence for Radio Button 24 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_24
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return
    
def get_STORED_sequence_25():
    """Get the STORED character sequence for Radio Button 25 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_25
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_26():
    """Get the STORED character sequence for Radio Button 26 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_26
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_27():
    """Get the STORED character sequence for Radio Button 27 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_27
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return
    
def get_STORED_sequence_28():
    """Get the STORED character sequence for Radio Button 28 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_28
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return
    
def get_STORED_sequence_29():
    """Get the STORED character sequence for Radio Button 29 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_29
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_30():
    """Get the STORED character sequence for Radio Button 30 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_30
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

def get_STORED_sequence_31():
    """Get the STORED character sequence for Radio Button 31 and pass to 'voice', the function that calls PYTTSX3."""
    if threading.active_count() <= 2:
        tx_data_input = RADIO_BUTTON_STORED_SEQUENCE_31
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
    else:
        return

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Set up the radio buttons for the STORED character sequences to be sent.
# Default is that NONE of the buttons are set.
sequence_selection_state = StringVar()
sequence_selection_state.set(None)

# Define a Tkinter StringVar to use to dynamically update the first radio button when desired.
# A regular Python variable will NOT work for Tkinter radio buttons.
RADIO_BUTTON_STORED_SEQUENCE_X = StringVar()
RADIO_BUTTON_STORED_SEQUENCE_X.set(dynamic_data_input)

# Display text to describe this group of radio buttons.
# Blank lines may be used above and below the text for proper formatting.
Label(rightwindow, text=" ", font=(RADIO_BUTTON_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 0).pack(pady = 3)
Label(rightwindow, text="Select Sentence to Transmit", font=(LABEL_TEXT_FONT), fg='blue',
      bg = BACKGROUND_COLOR, justify = LEFT, padx = 0).pack(pady = 0)
Label(rightwindow, text="<------>                                           Input File - SSB_Program_Data.py                                           <------>",
      font=(LABEL_TEXT_FONT), fg='blue',
      bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 5)
Label(rightwindow, text=" ", font=(RADIO_BUTTON_FONT), bg = BACKGROUND_COLOR, justify = RIGHT, padx = 0, pady = 0).pack(pady = RIGHT_WINDOW_PAD_Y)

# The multiple IF statements below look a bit odd, but they were put in to account for ANY number of
# radio buttons to be used, up to 32.  If more than 32 are desired, then the IF statements will need
# to be extended accordingly.  But if 32 or less are defined in the data file, this structure allows
# only that number to be used, and the right side of the GUI window will be sized accordingly.  This
# is legacy code in this application, and was more important in previous versions where vertical size
# mattered more.  But the structure is maintained, as it works and is extensible for possible future
# needs with vertical sizing.
if radio_button_count >= 1:
    # Uncomment the line below if it is not desired to use dynamic updates for the text of the
    # first radio button, and change the 'textvariable' input to match the others below.
    #RADIO_BUTTON_STORED_SEQUENCE_0 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_0

    # The text color for the radio buttons is defined as follows in the input data file.
    # Each radio button is handled the same below.
    FOREGROUND_0 = SSB_Program_Data.FOREGROUND_COLOR_0

    # The first radio button is unique in that it can be defined both from the input data file and
    # via text input and locked in with the 'Change Dynamic Text' button.  As such, this button uses
    # a 'textvariable' descriptor and not the 'text' version used in the others.  This is updated with
    # the Tkinter StringVar as shown, allowing the text to be updated dynamically.
    Radiobutton(rightwindow, textvariable=RADIO_BUTTON_STORED_SEQUENCE_X, font=(RADIO_BUTTON_FONT_DYNAMIC), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=0, fg=FOREGROUND_0, command = lambda: get_STORED_sequence_0()).pack(side = 'top', anchor = 'w')

if radio_button_count >= 2:
    RADIO_BUTTON_STORED_SEQUENCE_1 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_1
    FOREGROUND_1 = SSB_Program_Data.FOREGROUND_COLOR_1

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_1, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=1, fg=FOREGROUND_1, command = get_STORED_sequence_1).pack(side = 'top', anchor = 'w')

if radio_button_count >= 3:
    RADIO_BUTTON_STORED_SEQUENCE_2 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_2
    FOREGROUND_2 = SSB_Program_Data.FOREGROUND_COLOR_2

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_2, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=2, fg=FOREGROUND_2, command = get_STORED_sequence_2).pack(side = 'top', anchor = 'w')

if radio_button_count >= 4:
    RADIO_BUTTON_STORED_SEQUENCE_3 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_3
    FOREGROUND_3 = SSB_Program_Data.FOREGROUND_COLOR_3

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_3, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=3, fg=FOREGROUND_3, command = get_STORED_sequence_3).pack(side = 'top', anchor = 'w')

if radio_button_count >= 5:
    RADIO_BUTTON_STORED_SEQUENCE_4 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_4
    FOREGROUND_4 = SSB_Program_Data.FOREGROUND_COLOR_4

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_4, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=4, fg=FOREGROUND_4, command = get_STORED_sequence_4).pack(side = 'top', anchor = 'w')

if radio_button_count >= 6:
    RADIO_BUTTON_STORED_SEQUENCE_5 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_5
    FOREGROUND_5 = SSB_Program_Data.FOREGROUND_COLOR_5

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_5, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=5, fg=FOREGROUND_5, command = get_STORED_sequence_5).pack(side = 'top', anchor = 'w')

if radio_button_count >= 7:
    RADIO_BUTTON_STORED_SEQUENCE_6 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_6
    FOREGROUND_6 = SSB_Program_Data.FOREGROUND_COLOR_6

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_6, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=6, fg=FOREGROUND_6, command = get_STORED_sequence_6).pack(side = 'top', anchor = 'w')

if radio_button_count >= 8:
    RADIO_BUTTON_STORED_SEQUENCE_7 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_7
    FOREGROUND_7 = SSB_Program_Data.FOREGROUND_COLOR_7

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_7, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=7, fg=FOREGROUND_7, command = get_STORED_sequence_7).pack(side = 'top', anchor = 'w')

if radio_button_count >= 9:
    RADIO_BUTTON_STORED_SEQUENCE_8 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_8
    FOREGROUND_8 = SSB_Program_Data.FOREGROUND_COLOR_8

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_8, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=8, fg=FOREGROUND_8, command = get_STORED_sequence_8).pack(side = 'top', anchor = 'w')

if radio_button_count >= 10:
    RADIO_BUTTON_STORED_SEQUENCE_9 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_9
    FOREGROUND_9 = SSB_Program_Data.FOREGROUND_COLOR_9

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_9, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=9, fg=FOREGROUND_9, command = get_STORED_sequence_9).pack(side = 'top', anchor = 'w')

if radio_button_count >= 11:
    RADIO_BUTTON_STORED_SEQUENCE_10 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_10
    FOREGROUND_10 = SSB_Program_Data.FOREGROUND_COLOR_10

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_10, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=10, fg=FOREGROUND_10, command = get_STORED_sequence_10).pack(side = 'top', anchor = 'w')

if radio_button_count >= 12:
    RADIO_BUTTON_STORED_SEQUENCE_11 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_11
    FOREGROUND_11 = SSB_Program_Data.FOREGROUND_COLOR_11

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_11, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=11, fg=FOREGROUND_11, command = get_STORED_sequence_11).pack(side = 'top', anchor = 'w')

if radio_button_count >= 13:
    RADIO_BUTTON_STORED_SEQUENCE_12 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_12
    FOREGROUND_12 = SSB_Program_Data.FOREGROUND_COLOR_12

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_12, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=12, fg=FOREGROUND_12,  command = get_STORED_sequence_12).pack(side = 'top', anchor = 'w')

if radio_button_count >= 14:
    RADIO_BUTTON_STORED_SEQUENCE_13 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_13
    FOREGROUND_13 = SSB_Program_Data.FOREGROUND_COLOR_13

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_13, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=13, fg=FOREGROUND_13, command = get_STORED_sequence_13).pack(side = 'top', anchor = 'w')

if radio_button_count >= 15:
    RADIO_BUTTON_STORED_SEQUENCE_14 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_14
    FOREGROUND_14 = SSB_Program_Data.FOREGROUND_COLOR_14

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_14, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=14, fg=FOREGROUND_14, command = get_STORED_sequence_14).pack(side = 'top', anchor = 'w')

if radio_button_count >= 16:
    RADIO_BUTTON_STORED_SEQUENCE_15 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_15
    FOREGROUND_15 = SSB_Program_Data.FOREGROUND_COLOR_15

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_15, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=15, fg=FOREGROUND_15, command = get_STORED_sequence_15).pack(side = 'top', anchor = 'w')

if radio_button_count >= 17:
    RADIO_BUTTON_STORED_SEQUENCE_16 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_16
    FOREGROUND_16 = SSB_Program_Data.FOREGROUND_COLOR_16

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_16, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=16, fg=FOREGROUND_16, command = get_STORED_sequence_16).pack(side = 'top', anchor = 'w')

if radio_button_count >= 18:
    RADIO_BUTTON_STORED_SEQUENCE_17 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_17
    FOREGROUND_17 = SSB_Program_Data.FOREGROUND_COLOR_17

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_17, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=17, fg=FOREGROUND_17, command = get_STORED_sequence_17).pack(side = 'top', anchor = 'w')

if radio_button_count >= 19:
    RADIO_BUTTON_STORED_SEQUENCE_18 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_18
    FOREGROUND_18 = SSB_Program_Data.FOREGROUND_COLOR_18

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_18, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=18, fg=FOREGROUND_18, command = get_STORED_sequence_18).pack(side = 'top', anchor = 'w')

if radio_button_count >= 20:
    RADIO_BUTTON_STORED_SEQUENCE_19 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_19
    FOREGROUND_19 = SSB_Program_Data.FOREGROUND_COLOR_19

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_19, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=19, fg=FOREGROUND_19, command = get_STORED_sequence_19).pack(side = 'top', anchor = 'w')
    
if radio_button_count >= 21:
    RADIO_BUTTON_STORED_SEQUENCE_20 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_20
    FOREGROUND_20 = SSB_Program_Data.FOREGROUND_COLOR_20

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_20, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=20, fg=FOREGROUND_20, command = get_STORED_sequence_20).pack(side = 'top', anchor = 'w')

if radio_button_count >= 22:
    RADIO_BUTTON_STORED_SEQUENCE_21 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_21
    FOREGROUND_21 = SSB_Program_Data.FOREGROUND_COLOR_21

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_21, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=21, fg=FOREGROUND_21, command = get_STORED_sequence_21).pack(side = 'top', anchor = 'w')

if radio_button_count >= 23:
    RADIO_BUTTON_STORED_SEQUENCE_22 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_22
    FOREGROUND_22 = SSB_Program_Data.FOREGROUND_COLOR_22

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_22, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=22, fg=FOREGROUND_22, command = get_STORED_sequence_22).pack(side = 'top', anchor = 'w')

if radio_button_count >= 24:
    RADIO_BUTTON_STORED_SEQUENCE_23 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_23
    FOREGROUND_23 = SSB_Program_Data.FOREGROUND_COLOR_23

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_23, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=23, fg=FOREGROUND_23, command = get_STORED_sequence_23).pack(side = 'top', anchor = 'w')

if radio_button_count >= 25:
    RADIO_BUTTON_STORED_SEQUENCE_24 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_24
    FOREGROUND_24 = SSB_Program_Data.FOREGROUND_COLOR_24

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_24, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=24, fg=FOREGROUND_24, command = get_STORED_sequence_24).pack(side = 'top', anchor = 'w')
    
if radio_button_count >= 26:
    RADIO_BUTTON_STORED_SEQUENCE_25 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_25
    FOREGROUND_25 = SSB_Program_Data.FOREGROUND_COLOR_25

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_25, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=25, fg=FOREGROUND_25, command = get_STORED_sequence_25).pack(side = 'top', anchor = 'w')

if radio_button_count >= 27:
    RADIO_BUTTON_STORED_SEQUENCE_26 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_26
    FOREGROUND_26 = SSB_Program_Data.FOREGROUND_COLOR_26

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_26, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=26, fg=FOREGROUND_26, command = get_STORED_sequence_26).pack(side = 'top', anchor = 'w')

if radio_button_count >= 28:
    RADIO_BUTTON_STORED_SEQUENCE_27 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_27
    FOREGROUND_27 = SSB_Program_Data.FOREGROUND_COLOR_27

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_27, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=27, fg=FOREGROUND_27, command = get_STORED_sequence_27).pack(side = 'top', anchor = 'w')
    
if radio_button_count >= 29:
    RADIO_BUTTON_STORED_SEQUENCE_28 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_28
    FOREGROUND_28 = SSB_Program_Data.FOREGROUND_COLOR_28

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_28, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=28, fg=FOREGROUND_28, command = get_STORED_sequence_28).pack(side = 'top', anchor = 'w')
    
if radio_button_count >= 30:
    RADIO_BUTTON_STORED_SEQUENCE_29 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_29
    FOREGROUND_29 = SSB_Program_Data.FOREGROUND_COLOR_29

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_29, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=29, fg=FOREGROUND_29, command = get_STORED_sequence_29).pack(side = 'top', anchor = 'w')

if radio_button_count >= 31:
    RADIO_BUTTON_STORED_SEQUENCE_30 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_30
    FOREGROUND_30 = SSB_Program_Data.FOREGROUND_COLOR_30

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_30, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=30, fg=FOREGROUND_30, command = get_STORED_sequence_30).pack(side = 'top', anchor = 'w')

if radio_button_count == 32:
    RADIO_BUTTON_STORED_SEQUENCE_31 = SSB_Program_Data.RADIO_BUTTON_FILED_SEQUENCE_31
    FOREGROUND_31 = SSB_Program_Data.FOREGROUND_COLOR_31

    Radiobutton(rightwindow, text=RADIO_BUTTON_STORED_SEQUENCE_31, font=(RADIO_BUTTON_FONT), pady = 5,
    bg = BACKGROUND_COLOR, padx = RADIO_BUTTON_HORIZONTAL_PAD, variable=sequence_selection_state, highlightthickness = 0,
    value=31, fg=FOREGROUND_31, command = get_STORED_sequence_31).pack(side = 'top', anchor = 'w')

Label(rightwindow, text=" ", font=(RADIO_BUTTON_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 0).pack(pady = 5)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Display text to describe the text entry buffer.
# Blank lines may be used above and below the text for proper formatting.
Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 5)
Label(leftwindow, text="Type Text in Either Box Below & Click the", font=(LABEL_TEXT_FONT), fg='blue',
      bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)
Label(leftwindow, text="Appropriate Button When Ready to Transmit", font=(LABEL_TEXT_FONT), fg='blue',
      bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)
Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Function to input characters to transmit into the GUI
def tx_buffer_click():
    """Get characters to transmit from the first text box into the GUI buffer, then transfer into the main
    processing function 'voice' by clicking on the "Transmit Text Buffer" button."""
    # tx_data_input is the character string typed into the first GUI buffer and needs to be passed to the
    # main processing fuction, voice.
    if threading.active_count() <= 2:
        tx_data_input = tx.get()
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()
        # After the 'Transmit Text Buffer A' button is pressed, clear out the input text buffer.
        tx.delete(0, END)
    else:
        return

def ty_buffer_click():
    """Get characters to transmit from the phonetic text box into the GUI buffer, then transfer into the phonetic
    text processing dictionary by clicking on the "Transmit Phonetic Text Buffer" button."""
    # phonetic_data_input is the character string typed into the second GUI buffer and needs to be passed to the
    # phonetic character dictionary.
    global phonetic_word
    if threading.active_count() <= 2:
        phonetic_data_input = ty.get()
        # Allow the "Transmit Phonetic Text Buffer" button to be pressed when no characters are in the buffer.
        phonetic_word = ' '
        
        t1 = threading.Thread(target=dictionary, args=[phonetic_data_input])
        t1.start()
    # Do not clear out the phonetic text buffer, as it is likely to be used multiple times.
    else:
        return

def tz_buffer_click():
    """Get characters from the third text box to modify the first radio button, then transfer into the radio
    button by clicking on "Change Dynamic Text Radio Button"."""
    global dynamic_data_input

    dynamic_data_input = tz.get()
    # The line below updates the Tkinter StringVar associated with dynamic text input for the
    # first radio button when the 'Change Dynamic Text' button is pressed.
    RADIO_BUTTON_STORED_SEQUENCE_X.set(dynamic_data_input)
    # After the 'Change Dynamic Text' button is pressed, clear out the input text buffer.
    tz.delete(0, END)
    
    return

def tz2_buffer_click():
    """This is where the "Increment QSO #" button is implemented.  It is used for SSB contests where the format is
    '5 9, 001, thank you' and the serial number needs to be updated after a successful QSO."""
    character_buffer_clear = True
    character_buffer_a = False
    global dynamic_data_input
    
    match = re.search(' (\d+)[^\d]*$',dynamic_data_input)
    if not match:
        print ('No Match!')
        return dynamic_data_input
    new_number = int(match.group(1)) + 1
    dynamic_data_input = re.sub('\d+([^\d]*)$', '%d\g<1>' % new_number, dynamic_data_input)
        
    RADIO_BUTTON_STORED_SEQUENCE_X.set(dynamic_data_input)
    
    return

def tz3_buffer_click():
    """This is where the "Repeat QSO #" button is implemented.  It is used if the op on the other end requests a
    of the QSO serial number only."""
    character_buffer_clear = True
    character_buffer_a = False
    global dynamic_data_input
    global tx_data_input
    
    match = re.search(' (\d+)[^\d]*$',dynamic_data_input)
    if not match:
        print ('No Match!')
        return
        
    # Append the word 'number' to the repeated serial number, with added delay for better understandability in
    # the speech synthesizer.
    tx_data_input = 'number, , , , ' + match.group(1)
            
    if threading.active_count() <= 2:
        t1 = threading.Thread(target=voice, args=[tx_data_input])
        t1.start()        
    else:
        return   

    return

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Set the first character typing box length in number of characters defined as a CONSTANT.
# Use this same length for both character input boxes.
tx = Entry(leftwindow, width = ENTRY_BOX_CHARACTER_WIDTH, font=(ENTRY_BOX_TEXT_FONT), bd = 6, bg = 'gray95')
tx.pack(padx = 20, pady = 10)

# Get the characters typed into the first GUI buffer and place them into the input buffer.
# Set up the button for the first character input box that will transmit the characters typed in.
TX_Button = Button(leftwindow, text = "Transmit Text Buffer", font=(ENABLE_BUTTON_FONT), fg='green', command = tx_buffer_click, bd = 4)
TX_Button.pack(pady = 20)
Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)
#Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)

# Set the second character typing box length in number of characters defined as a CONSTANT.
# Use this same length for both character input boxes.
ty = Entry(leftwindow, width = ENTRY_BOX_CHARACTER_WIDTH, font=(ENTRY_BOX_TEXT_FONT), bd = 6, bg = 'gray95')
ty.pack(padx = 20, pady = 10)

# Get the characters typed into the second GUI buffer and place them into the input buffer.
# Set up the button for the second character input box that will transmit the characters typed in.
TX_Button = Button(leftwindow, text = "Transmit Phonetic Text Buffer", font=(ENABLE_BUTTON_FONT), fg='green', command = ty_buffer_click, bd = 4)
TX_Button.pack(pady = 20)

Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 4)

Label(leftwindow, text="Type Text in the Box Below & Click the", font=(LABEL_TEXT_FONT), fg='blue',
      bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)
Label(leftwindow, text="Button to Change the First Radio Button Text", font=(LABEL_TEXT_FONT), fg='blue',
      bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)

# Set the third character typing box length in number of characters defined as a CONSTANT.
# Use this same length as the previously defined character input boxes.
tz= Entry(leftwindow, width = ENTRY_BOX_CHARACTER_WIDTH, font=(SECONDARY_ENTRY_BOX_TEXT_FONT), bd = 6, bg = 'gray95')
tz.pack(padx = 20, pady = 25)

# Get the characters typed into the third box and use later as a Tkinter StringVar for updating the first radio button if desired.
TZ_Button = Button(leftwindow, text = "Change Dynamic Text Radio Button", font=(ENABLE_BUTTON_FONT), fg='green', command = lambda: tz_buffer_click(), bd = 4)
TZ_Button.pack(pady = 5)

#Define the "Increment QSO #" button.
TZ2_Button = Button(leftwindow, text = "Increment QSO #", font=(ENABLE_BUTTON_FONT), fg='purple', command = lambda: tz2_buffer_click(), bd = 4)
TZ2_Button.pack(pady = 5)

#Define the "Repeat QSO #" button.
TZ3_Button = Button(leftwindow, text = "Repeat QSO #", font=(ENABLE_BUTTON_FONT), fg='brown', command = lambda: tz3_buffer_click(), bd = 4)
TZ3_Button.pack(pady = 5)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Execute this code when the program is first opened to correctly set up the radio button variables.
radio_button_state_1 = StringVar()
radio_button_state_2 = StringVar()
radio_button_state_3 = StringVar()
radio_button_state_4 = StringVar()

radio_button_rate_state_1 = StringVar()
radio_button_rate_state_2 = StringVar()
radio_button_rate_state_3 = StringVar()

# Set the default voice model to "US English Male Voice" with a "0" here.
radio_button_state_1.set(0)

# Set the default voice rate to "Normal Voice Rate" with a "1" here.
radio_button_rate_state_1.set(1)

# Set up the voice selection buttons.
# Draw a staight line across the left box between the 3 text boxes and the parameter choices.
Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)

Label(leftwindow, text="__________________________________________________", font=(LABEL_TEXT_FONT),
      bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)

Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)
Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)
Label(leftwindow, text="Choose a Voice Model", font=(LABEL_TEXT_FONT), fg='blue',
      bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)
Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), fg='blue',
      bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)

Radiobutton(leftwindow, text=VOICE_SELECTION_1, font=(RADIO_BUTTON_FONT), bg = BACKGROUND_COLOR, highlightthickness = 0,
            padx = VOICE_RADIO_BUTTON_HORIZONTAL_PAD_MODE, pady = 2, variable=radio_button_state_1, value=0).pack(side = 'top', anchor = 'w')
Radiobutton(leftwindow, text=VOICE_SELECTION_2, font=(RADIO_BUTTON_FONT), bg = BACKGROUND_COLOR, highlightthickness = 0,
            padx = VOICE_RADIO_BUTTON_HORIZONTAL_PAD_MODE, variable=radio_button_state_1, value=1).pack(side = 'top', anchor = 'w')
Radiobutton(leftwindow, text=VOICE_SELECTION_3, font=(RADIO_BUTTON_FONT), bg = BACKGROUND_COLOR, highlightthickness = 0,
            padx = VOICE_RADIO_BUTTON_HORIZONTAL_PAD_MODE, pady = 2, variable=radio_button_state_1, value=2).pack(side = 'top', anchor = 'w')
Radiobutton(leftwindow, text=VOICE_SELECTION_4, font=(RADIO_BUTTON_FONT), bg = BACKGROUND_COLOR, highlightthickness = 0,
            padx = VOICE_RADIO_BUTTON_HORIZONTAL_PAD_MODE, variable=radio_button_state_1, value=3).pack(side = 'top', anchor = 'w')

Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), fg='blue',
      bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

Label(leftwindow, text="Select the Desired Voice Rate", font=(LABEL_TEXT_FONT), fg='blue',
      bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)
Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)

Radiobutton(leftwindow, text=VOICE_RATE_SELECTION_1, font=(RADIO_BUTTON_FONT), bg = BACKGROUND_COLOR, highlightthickness = 0,
            padx = VOICE_RADIO_BUTTON_HORIZONTAL_PAD_MODE, pady = 2, variable=radio_button_rate_state_1, value=0).pack(side = 'top', anchor = 'w')
Radiobutton(leftwindow, text=VOICE_RATE_SELECTION_2, font=(RADIO_BUTTON_FONT), bg = BACKGROUND_COLOR, highlightthickness = 0,
            padx = VOICE_RADIO_BUTTON_HORIZONTAL_PAD_MODE, variable=radio_button_rate_state_1, value=1).pack(side = 'top', anchor = 'w')
Radiobutton(leftwindow, text=VOICE_RATE_SELECTION_3, font=(RADIO_BUTTON_FONT), bg = BACKGROUND_COLOR, highlightthickness = 0,
            padx = VOICE_RADIO_BUTTON_HORIZONTAL_PAD_MODE, pady = 2, variable=radio_button_rate_state_1, value=2).pack(side = 'top', anchor = 'w')

Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)
Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)
Label(leftwindow, text=" ", font=(LABEL_TEXT_FONT), bg = BACKGROUND_COLOR, justify = LEFT, padx = 20).pack(pady = 0)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

phonetic_data_input = ' '

# Define the alphabet characters here, A-Z.
# All upper case characters are the same as their lower case equivalents, ie 'a' and 'A'.
# The functions in this section use the dictionary to translate each character in the Transmit phonetic
# Text Buffer to the appropriate phonetic word, and then concatenate into a separate text buffer
# for input to the text-to-speech synthesizer.  A separate buffer is used for speed, as the PYTTSX3
# module only needs to be opened and closed once per sequence.  Otherwise there is a very noticable
# delay between phonetic words.  There is an additional space character after every word for proper
# interpretation of each individual word by the synthesizer.

def letter_aA():
    global phonetic_word
    phonetic_word += 'alpha '
    
def letter_bB():
    global phonetic_word
    phonetic_word += 'bravo '
    
def letter_cC():
    global phonetic_word
    phonetic_word += 'charlie '

def letter_dD():
    global phonetic_word
    phonetic_word += 'delta '
    
def letter_eE():
    global phonetic_word
    phonetic_word += 'echo '
    
def letter_fF():
    global phonetic_word
    phonetic_word += 'foxtrot '
    
def letter_gG():
    global phonetic_word
    phonetic_word += 'golf '
    
def letter_hH():
    global phonetic_word
    phonetic_word += 'hotel '
    
def letter_iI():
    global phonetic_word
    phonetic_word += 'india '

def letter_jJ():
    global phonetic_word
    phonetic_word += 'juliette '
    
def letter_kK():
    global phonetic_word
    phonetic_word += 'kilo '

def letter_lL():
    global phonetic_word
    phonetic_word += 'lima '

def letter_mM():
    global phonetic_word
    phonetic_word += 'mike '

def letter_nN():
    global phonetic_word
    phonetic_word += 'november '

def letter_oO():
    global phonetic_word
    phonetic_word += 'oscar '
    
def letter_pP():
    global phonetic_word
    phonetic_word += 'papa '
    
def letter_qQ():
    global phonetic_word
    phonetic_word += 'quebec '
    
def letter_rR():
    global phonetic_word
    phonetic_word += 'romeo '

def letter_sS():
    global phonetic_word
    phonetic_word += 'sierra '

def letter_tT():
    global phonetic_word
    phonetic_word += 'tango '

def letter_uU():
    global phonetic_word
    phonetic_word += 'uniform '
    
def letter_vV():
    global phonetic_word
    phonetic_word += 'victor '

def letter_wW():
    global phonetic_word
    phonetic_word += 'whiskey '
    
def letter_xX():
    global phonetic_word
    phonetic_word += 'xray '

def letter_yY():
    global phonetic_word
    phonetic_word += 'yankee '
    
def letter_zZ():
    global phonetic_word
    phonetic_word += 'zulu '

# Define the number characters here, 0-9

def number_0():
    global phonetic_word
    phonetic_word += 'zero '

def number_1():
    global phonetic_word
    phonetic_word += 'one '
    
def number_2():
    global phonetic_word
    phonetic_word += 'two '
    
def number_3():
    global phonetic_word
    phonetic_word += 'three '
    
def number_4():
    global phonetic_word
    phonetic_word += 'four '
    
def number_5():
    global phonetic_word
    phonetic_word += 'five '
    
def number_6():
    global phonetic_word
    phonetic_word += 'six '
    
def number_7():
    global phonetic_word
    phonetic_word += 'seven '

def number_8():
    global phonetic_word
    phonetic_word += 'eight '
    
def number_9():
    global phonetic_word
    phonetic_word += 'nine '
    
# Define the punctuation characters here.

def punctuation_dash():
    global phonetic_word
    phonetic_word += 'dash '

def punctuation_equals():
    global phonetic_word
    phonetic_word += 'equals '

def punctuation_period():
    global phonetic_word
    phonetic_word += 'dot '
    
def punctuation_comma():
    global phonetic_word
    phonetic_word += 'comma '
    
def punctuation_question_mark():
    global phonetic_word
    phonetic_word += ' '
    
def punctuation_slash():
    global phonetic_word
    phonetic_word += 'stroke '
    
def punctuation_ampersand():
    global phonetic_word
    phonetic_word += 'at '
    
def special_function_bar():
    global phonetic_word
    phonetic_word += 'bar '
    
def error_character():
    global phonetic_word
    phonetic_word += ' '
    
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
class TTSVoice:
    # Preload all 4 currently supported voice models in Piper TTS at program start to reduce latency.
    # The GUI will take approximately 4 times 2 seconds, or 8 seconds total, to load but that is OK.
    voicedir = "/home/radiopi1/QST/"
    
    model1 = voicedir+"en_US-ryan-low.onnx"
    model2 = voicedir+"en_US-amy-low.onnx"
    model3 = voicedir+"en_US-danny-low.onnx"
    model4 = voicedir+"en_US-kathleen-low.onnx"
    
    voice1 = PiperVoice.load(model1)
    voice2 = PiperVoice.load(model2)
    voice3 = PiperVoice.load(model3)
    voice4 = PiperVoice.load(model4)
    
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

def voice(tx_data_input):
    """This is the function that sets up and calls the Python voice synthesis package Piper TTS.
    It is called by any of the radio buttons with text from the file SSB_Program?Data.py and from
    the Transmit Text Buffer.
    """
    # Check to see if the Transmit Text Buffer has any text to convert to speech after clicking on
    # the "Transmit Text Buffer" button.  Otherwise, if the buffer is empty when the button is
    # clicked the program will hang after calling "engine".
        
    if tx_data_input:        
        # Query the radio buttons to choose the desired voice.
        temp_var_1 = str(radio_button_state_1.get())     
        # Query the radio buttons to choose the desired voice rate.
        temp_var_2 = str(radio_button_rate_state_1.get())
        
        if temp_var_2 == "0":
            length_scale_value = 0.5
        elif temp_var_2 == "1":
            length_scale_value = 1.0
        else:
            length_scale_value = 2.0

        # Turn on the microphone PTT signal to the radio.
        VOICE_SWITCH.on()
        
        text = tx_data_input

        ttsvoice = TTSVoice()

        if temp_var_1 == "0":
            stream = sd.OutputStream(samplerate=ttsvoice.voice1.config.sample_rate, channels=1, dtype='int16')
            stream.start()

            for audio_bytes in ttsvoice.voice1.synthesize_stream_raw(text, length_scale=length_scale_value):
                int_data = np.frombuffer(audio_bytes, dtype=np.int16)
                stream.write(int_data)
                
        elif temp_var_1 == "1":
            stream = sd.OutputStream(samplerate=ttsvoice.voice2.config.sample_rate, channels=1, dtype='int16')
            stream.start()

            for audio_bytes in ttsvoice.voice2.synthesize_stream_raw(text, length_scale=length_scale_value):
                int_data = np.frombuffer(audio_bytes, dtype=np.int16)
                stream.write(int_data)
                
        elif temp_var_1 == "2":
            stream = sd.OutputStream(samplerate=ttsvoice.voice3.config.sample_rate, channels=1, dtype='int16')
            stream.start()

            for audio_bytes in ttsvoice.voice3.synthesize_stream_raw(text, length_scale=length_scale_value):
                int_data = np.frombuffer(audio_bytes, dtype=np.int16)
                stream.write(int_data)
                
        else:
            stream = sd.OutputStream(samplerate=ttsvoice.voice4.config.sample_rate, channels=1, dtype='int16')
            stream.start()

            for audio_bytes in ttsvoice.voice4.synthesize_stream_raw(text, length_scale=length_scale_value):
                int_data = np.frombuffer(audio_bytes, dtype=np.int16)
                stream.write(int_data)   
        
        time.sleep(0.1)
        stream.stop()
        stream.close()
                        
        # Turn off the microphone PTT signal to the radio.
        time.sleep(0.1)
        VOICE_SWITCH.off()
                
    # If nothing is in the buffer, return without calling "engine".
    else:
        return
    
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
      
def voice_phonetic_word(phonetic_word):
    """This is the function that sets up and calls the Python voice synthesis package Piper TTS."""

    # Query the radio buttons to choose the desired voice.
    temp_var_1 = str(radio_button_state_1.get())     
    # Query the radio buttons to choose the desired voice rate.
    temp_var_2 = str(radio_button_rate_state_1.get())
    
    if temp_var_2 == "0":
        length_scale_value = 0.5
    elif temp_var_2 == "1":
        length_scale_value = 1.0
    else:
        length_scale_value = 2.0

    # Turn on the microphone PTT signal to the radio.
    VOICE_SWITCH.on()
        
    text = phonetic_word

    ttsvoice = TTSVoice()

    if temp_var_1 == "0":
        stream = sd.OutputStream(samplerate=ttsvoice.voice1.config.sample_rate, channels=1, dtype='int16')
        stream.start()

        for audio_bytes in ttsvoice.voice1.synthesize_stream_raw(text, length_scale=length_scale_value):
            int_data = np.frombuffer(audio_bytes, dtype=np.int16)
            stream.write(int_data)
                
    elif temp_var_1 == "1":
        stream = sd.OutputStream(samplerate=ttsvoice.voice2.config.sample_rate, channels=1, dtype='int16')
        stream.start()

        for audio_bytes in ttsvoice.voice2.synthesize_stream_raw(text, length_scale=length_scale_value):
            int_data = np.frombuffer(audio_bytes, dtype=np.int16)
            stream.write(int_data)
                
    elif temp_var_1 == "2":
        stream = sd.OutputStream(samplerate=ttsvoice.voice3.config.sample_rate, channels=1, dtype='int16')
        stream.start()

        for audio_bytes in ttsvoice.voice3.synthesize_stream_raw(text, length_scale=length_scale_value):
            int_data = np.frombuffer(audio_bytes, dtype=np.int16)
            stream.write(int_data)
                
    else:
        stream = sd.OutputStream(samplerate=ttsvoice.voice4.config.sample_rate, channels=1, dtype='int16')
        stream.start()

        for audio_bytes in ttsvoice.voice4.synthesize_stream_raw(text, length_scale=length_scale_value):
            int_data = np.frombuffer(audio_bytes, dtype=np.int16)
            stream.write(int_data)
    
    # Stop and close the output audio stream.
    time.sleep(0.1)
    stream.stop()
    stream.close()
    
    # Turn off the microphone PTT signal to the radio.
    time.sleep(0.1)
    VOICE_SWITCH.off()
        
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# This function is where the list processing is performed for all characters.
# The function operates on both the STORED radio button sequences, as well as
# characters typed into the text box via the keyboard.
def dictionary(phonetic_data_input):
    """ Main function that processes characters to be transmitted."""
    # Convert the input character string to individual characters for processing.
    phonetic_data_characters = list(phonetic_data_input)
    # Find out how many characters are in the line that need processing.
    length_of_input = len(phonetic_data_characters)
    # Start processing characters at position 0 of the character buffer list.
    length_of_input_start = 0

    global phonetic_word

    # Convert each input character to a phonetic equivalent one at a time.
    while length_of_input_start < length_of_input:

        # Define a dictionary with the keys to be the input character,
        # and the values to be the function "representation" to be called
        # to process the character.
        #
        # Cannot use a function call directly as a dictionary value or ALL
        # will be called sequentially when any key is accessed.
        #
        # All lowercase letters must be pared with an uppercase equivalent.

        switcher = {
            'a': letter_aA,
            'A': letter_aA,
            'b': letter_bB,
            'B': letter_bB,
            'c': letter_cC,
            'C': letter_cC,
            'd': letter_dD,
            'D': letter_dD,
            'e': letter_eE,
            'E': letter_eE,
            'f': letter_fF,
            'F': letter_fF,
            'g': letter_gG,
            'G': letter_gG,
            'h': letter_hH,
            'H': letter_hH,
            'i': letter_iI,
            'I': letter_iI,
            'j': letter_jJ,
            'J': letter_jJ,
            'k': letter_kK,
            'K': letter_kK,
            'l': letter_lL,
            'L': letter_lL,
            'm': letter_mM,
            'M': letter_mM,
            'n': letter_nN,
            'N': letter_nN,
            'o': letter_oO,
            'O': letter_oO,
            'p': letter_pP,
            'P': letter_pP,
            'q': letter_qQ,
            'Q': letter_qQ,
            'r': letter_rR,
            'R': letter_rR,
            's': letter_sS,
            'S': letter_sS,
            't': letter_tT,
            'T': letter_tT,
            'u': letter_uU,
            'U': letter_uU,
            'v': letter_vV,
            'V': letter_vV,
            'w': letter_wW,
            'W': letter_wW,
            'x': letter_xX,
            'X': letter_xX,
            'y': letter_yY,
            'Y': letter_yY,
            'z': letter_zZ,
            'Z': letter_zZ,

            #' ': word_end_space_processing,

            '0': number_0,
            '1': number_1,
            '2': number_2,
            '3': number_3,
            '4': number_4,
            '5': number_5,
            '6': number_6,
            '7': number_7,
            '8': number_8,
            '9': number_9,

            '.': punctuation_period,
            ',': punctuation_comma,
            '?': punctuation_question_mark,
            '/': punctuation_slash,
            '-': punctuation_dash,
            '=': punctuation_equals,
            '@': punctuation_ampersand,

            '|': special_function_bar
        }
        
        # Get the current character to be processed.
        phonetic_data_characters_current = phonetic_data_characters[length_of_input_start]
        # Use the current character as a key to the dictionary and name it as a placeholder.
        # A default value must be used for proper calling,
        # so use the error character for any character not in the dictionary.
        dictionary_value_function = switcher.get(phonetic_data_characters_current, error_character)
        # Convert the placeholder to a function call to process the current character.
        dictionary_value_function()
        # Increment to the next character in the character buffer list.
        length_of_input_start += 1
            
    voice_phonetic_word(phonetic_word)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# The method mainloop() is an infinite loop used to run the GUI application.
# An event is processed as long as the window is open and running via this method.

window.mainloop()

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
